package org.jpwh.test.conversation;

import org.jpwh.env.JPATest;
import org.jpwh.model.conversation.Image;
import org.jpwh.model.conversation.Item;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;

import java.io.*;

import static org.testng.Assert.assertEquals;

public class ExtendedPC extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("ConversationPU");
    }

    // TODO: Błąd dla MySQL https://hibernate.atlassian.net/browse/HHH-8402
    @Test(groups = {"H2", "ORACLE", "POSTGRESQL"})
    public void conversationCreateItem() throws Exception {
        Item item = new Item("Jakiś przedmiot");
        item.getImages().add(new Image("Foo", "foo.jpg", 800, 600));

        CreateEditItemService conversation = new CreateEditItemService();
        conversation.storeItem(item);
        conversation.commit();

        {
            UserTransaction tx = TM.getUserTransaction();
            try {
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                item = em.find(Item.class, item.getId());
                assertEquals(item.getName(), "Jakiś przedmiot");
                assertEquals(item.getImages().size(), 1);
            } finally {
                TM.rollback();
            }
        }
    }

    // TODO: Błąd dla MySQL https://hibernate.atlassian.net/browse/HHH-8402
    @Test(groups = {"H2", "ORACLE", "POSTGRESQL"})
    public void conversationEditItem() throws Exception {
        final TestData testData = storeItemImagesTestData();
        Long ITEM_ID = testData.getFirstId();

        // Pierwsze zdarzenie w konwersacji
        CreateEditItemService conversation = new CreateEditItemService();
        Item item = conversation.getItem(ITEM_ID);

        item.setName("Nowa nazwa");
        // Kontekst utrwalania nadal otwarty. Ładowanie kolekcji na żądanie!
        item.getImages().add(new Image("Foo", "foo.jpg", 800, 600));

        // Możliwe dodatkowe zdarzenia. Ładowanie potrzebnych danych do konwersacji...

        // Ostatnie zdarzenie w konwersacji
        conversation.commit(); // Synchronizacja i zamknięcie kontekstu utrwalania

        {
            UserTransaction tx = TM.getUserTransaction();
            try {
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                item = em.find(Item.class, item.getId());
                assertEquals(item.getName(), "Nowa nazwa");
                assertEquals(item.getImages().size(), 4);
            } finally {
                TM.rollback();
            }
        }
    }

    public class CreateEditItemService {

        final EntityManager em;

        /* 
         Ta klasa <code>CreateEditItemService</code> ma jeden egzemplarz na konwersację.
         Jest on powiązany z jednym rozszerzonym kontekstem utrwalania. 
         Tworzymy <code>EntityManager</code> w momencie tworzenia usługi, 
          kiedy usługa musi obsłużyć pierwsze zdarzenie konwersacji. 
         Klient wielokrotnie korzysta z usługi do czasu zakończenia konwersacji.
         W tym momencie nie jest uruchamiana żadna transakcja, jedynie kontekst utrwalania jest otwarty.
		 
         */
        public CreateEditItemService() throws Exception {
            em = JPA.createEntityManager();
        }

        /* 
           Ta metoda obsługuje zdarzenie "załaduj <code>Item</code> do edycji". 
           Za pomocą obiektu <code>EntityManager</code> pobieramy egzemplarz <code>Item</code> 
           według identyfikatora i zwracamy go do klienta. Ponieważ w momencie zwrócenia sterowania przez metodę 
           kontekst utrwalania jest nadal otwarty , to jeśli zachodzi potrzeba klient może uzyskać dostęp 
           do kolekcji <code>Item#images</code>. Kolekcja jest inicjowana na żądanie. 
           Nie trzeba w tym miejscu jej wstępnie ładować..
         */
        protected Item getItem(Long itemId) {
            return em.find(Item.class, itemId);
        }

        /* 
            Ta metoda obsługuje zdarzenie "zapisz <code>Item</code> po edycji". 
            Klient musi ją wywołać tylko w celu zapisania nowego egzemplarza <code>Item</code> 
            w stanie przejściowym.  Kontekst utrwalania automatycznie wykrywa modyfikacje 
            wszystkich załadowanych i utrwalonych (zarządzanych) egzemplarzy <code>Item</code> 
            w momencie synchronizacji.
         */
        protected void storeItem(Item item) {
            em.persist(item);
        }

        /* 
           Część kontraktu usługi polega na tym, że klient musi wywołać tę metodę <code>commit()</code> 
           w czasie, kiedy konwersacja powinna zakończyć się powodzeniem.Ta metoda łączy obiekt 
           <code>EntityManager</code> z transakcją systemową, synchronizuje kontekst utrwalania, 
            zatwierdza transakcję, po czym zamyka obiekt <code>EntityManager</code>.
         */
        protected void commit() throws Exception {
            try {
                UserTransaction tx = TM.getUserTransaction();
                tx.begin();
                em.joinTransaction();
                tx.commit();
            } finally {
                em.close();
            }
        }
    }

    /* ################################################################################### */

    public TestData storeItemImagesTestData() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();
        Long[] ids = new Long[1];
        Item item = new Item();
        item.setName("Jakiś przedmiot");
        em.persist(item);
        ids[0] = item.getId();
        for (int i = 1; i <= 3; i++) {
            item.getImages().add(
                new Image("Zdjęcie " + i, "image" + i + ".jpg", 640, 480));
        }
        tx.commit();
        em.close();
        return new TestData(ids);
    }


    protected byte[] serialize(Object o) throws Exception {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutput out = null;
        try {
            out = new ObjectOutputStream(bos);
            out.writeObject(o);
            return bos.toByteArray();
        } finally {
            if (out != null)
                out.close();
            bos.close();
        }
    }

    protected Object deserialize(byte[] bytes) throws Exception {
        ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
        ObjectInput in = null;
        try {
            in = new ObjectInputStream(bis);
            return in.readObject();
        } finally {
            bis.close();
            if (in != null)
                in.close();
        }
    }

}
