package org.jpwh.test.concurrency;

import org.jpwh.env.JPATest;
import org.jpwh.model.concurrency.version.Item;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNotNull;

public class NonTransactional extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("ConcurrencyVersioningPU");
    }

    // TODO: Zawodzi dla MySQL https://hibernate.atlassian.net/browse/HHH-8402
    @Test(groups = {"H2", "ORACLE", "POSTGRESQL"})
    public void autoCommit() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        Long ITEM_ID;
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();
            Item someItem = new Item("Pierwotna nazwa");
            em.persist(someItem);
            tx.commit();
            em.close();
            ITEM_ID = someItem.getId();
        } finally {
            TM.rollback();
        }

        {
            /* 
               Żadna transakcja nie jest aktywna w momencie stworzenia obiektu <code>EntityManager</code>. Kontekst
               utrwalania jest teraz w specjalnym trybie <em>niezsynchronizowanym</em>. Hibernate
               nie spowoduje automatycznej synchronizacji kontekstu w dowolnym momencie.
             */
            EntityManager em = JPA.createEntityManager();

            /* 
               Możemy sięgnąć do bazy danych w celu odczytania danych. Ta operacja spowoduje uruchomienie
               instrukcji <code>SELECT</code>, która zostanie wysłana do bazy danych w trybie autozatwierdzania.
             */
            Item item = em.find(Item.class, ITEM_ID);
            item.setName("Nowa nazwa");

            /* 
               W momencie uruchamiania zapytania zwykle Hibernate synchronizuje kontekst utrwalania.
               . Ponieważ jednak kontekst jest <em>niezsynchronizowany</em>,
               to synchronizacja nie nastąpi i zapytanie zwróci starą, oryginalną wartość
               z bazy danych. Zapytania z wynikami skalarnymi nie są powtarzalne. Uzyskamy wartości, które
               występują w bazie danych i zostały przekazane do Hibernate w
               obiekcie <code>ResultSet</code>. Zwróćmy uwagę, że to nie jest powtarzalny odczyt nawet wtedy, gdy
               będziemy w trybie <em>zsynchronizowanym</em>.
             */
            assertEquals(
                em.createQuery("select i.name from Item i where i.id = :id)")
                    .setParameter("id", ITEM_ID).getSingleResult(),
                "Pierwotna nazwa"
            );

            /* 
               Pobranie zarządzanego egzemplarza encji wymaga wyszukiwania podczas 
               marshalingu zbioru wyników w bieżącym kontekście utrwalania. Załadowany
               egzemplarz <code>Item</code> ze zmienioną nazwą zostanie zwrócony z
               kontekstu utrwalania. Wartości z bazy danych 
               będą zignorowane. To jest powtarzalny odczyt egzemplarza encji
               nawet bez transakcji systemowej.
             */
            assertEquals(
                ((Item) em.createQuery("select i from Item i where i.id = :id)")
                    .setParameter("id", ITEM_ID).getSingleResult()).getName(),
                "Nowa nazwa"
            );

            /* 
               Jeśli spróbujemy ręcznie zsynchronizować kontekst za pomocą operacji flush tak, aby
               zapisać nowy obiekt <code>Item#name</code>, to Hibernate zgłosi wyjątek
               <code>javax.persistence.TransactionRequiredException</code>. Nie można
               uruchomić instrukcji <code>UPDATE</code> w trybie
               <em>niezsynchronizowanym</em>, ponieważ nie byłoby możliwości cofnięcia zmiany.
            */
            // em.flush();

            /* 
               Możemy cofnąć wprowadzoną zmianę, za pomocą metody <code>refresh()</code>.
               Metoda ta ładuje bieżący stan <code>Item</code> z bazy danych
               i nadpisuje zmiany wprowadzone w pamięci.
             */
            em.refresh(item);
            assertEquals(item.getName(), "Pierwotna nazwa");

            em.close();
        }

        {
            EntityManager em = JPA.createEntityManager();

            Item newItem = new Item("Nowy przedmiot");
            /* 
               W celu zapisania egzemplarza encji w stanie przejściowym możemy, w przypadku niezsynchronizowanego kontekstu utrwalania, wywołać metodę <code>persist()</code>.
               Hibernate pobiera tylko nową nową wartość identyfikatora, zwykle poprzez wywołanie sekwencji w bazie danych, i przypisuje uzyskaną wartość do egzemplarza.
               Egzemplarz jest teraz w stanie trwałym w tym kontekście, ale instrukcja SQL <code>INSERT</code> nie została wykonana. 
               Zwróćmy uwagę, że to jest możliwe tylko z generatorami identyfikatorów <em>pre-insert</em>; zobacz <a href="#GeneratorStrategies"/>.
            */
            em.persist(newItem);
            assertNotNull(newItem.getId());

            /* 
               Gdy jesteśmy gotowi do zapamiętania zmian, możemy scalić kontekst utrwalania z wykorzystaniem transakcji. 
               Synchronizacja i operacja flush są wykonywane tak, jak zwykle — w momencie zatwierdzania transakcji. 
               Hibernate zapisuje do bazy danych wszystkie operacje umieszczone w kolejce.
             */
            tx.begin();
            if (!em.isJoinedToTransaction())
                em.joinTransaction();
            tx.commit(); // Flush!
            em.close();
        }

        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();
            assertEquals(em.find(Item.class, ITEM_ID).getName(), "Pierwotna nazwa");
            assertEquals(em.createQuery("select count(i) from Item i)").getSingleResult(), 2l);
            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }

        {
            EntityManager tmp = JPA.createEntityManager();
            Item detachedItem = tmp.find(Item.class, ITEM_ID);
            tmp.close();

            detachedItem.setName("Nowa nazwa");
            EntityManager em = JPA.createEntityManager();

            Item mergedItem = em.merge(detachedItem);

            tx.begin();
            em.joinTransaction();
            tx.commit(); // Flush!
            em.close();
        }

        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();
            assertEquals(em.find(Item.class, ITEM_ID).getName(), "Nowa nazwa");
            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }

        {
            EntityManager em = JPA.createEntityManager();

            Item item = em.find(Item.class, ITEM_ID);
            em.remove(item);

            tx.begin();
            em.joinTransaction();
            tx.commit(); // Flush!
            em.close();
        }

        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();
            assertEquals(em.createQuery("select count(i) from Item i)").getSingleResult(), 1l);
            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }

}

