package org.jpwh.env;

import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;

import java.io.*;

/**
 * Uruchamia i zatrzymuje środowisko JPA przed (po) klasie testu.
 * <p>
 * Należy stworzyć podklasę w celu napisania testów jednostkowych. Skorzystaj z <code>EntityManagerFactory</code>
 * za pomocą odwołania {@link JPATest#JPA} i stwórz egzemplarze klasy <code>EntityManager</code>.
 * </p>
 * <p>
 * Usuwa i tworzy schemat bazy danych SQL dla jednostki utrwalania przed i po
 * każdej metodzie testu. To oznacza, że baza danych będzie czyszczona dla każdej metody testu.
 * </p>
 * <p>
 * Przesłoń metodę {@link #configurePersistenceUnit} w celu podania własnej
 * nazwy jednostki utrwalania, albo dodatkowe nazwy plików <code>hbm.xml</code> do załadowania na potrzeby
 * klasy testu.
 * </p>
 * <p>
 * Przesłoń metodę {@link #afterJPABootstrap()} w celu uruchomienia operacji po 
 * metodzie testu, ale zanim obiekt <code>EntityManagerFactory</code> będzie gotowy. W tym momencie
 * możesz stworzyć obiekt <code>EntityManager</code> lub <code>Session#doWork(JDBC)</code>. Jeżeli
 * jest potrzebne czyszczenie, przesłoń metodę {@link #beforeJPAClose()}.
 * </p>
 */
public class JPATest extends TransactionManagerTest {

    public String persistenceUnitName;
    public String[] hbmResources;
    public JPASetup JPA;

    @BeforeClass
    public void beforeClass() throws Exception {
        configurePersistenceUnit();
    }

    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit(null);
    }

    public void configurePersistenceUnit(String persistenceUnitName,
                                         String... hbmResources) throws Exception {
        this.persistenceUnitName = persistenceUnitName;
        this.hbmResources = hbmResources;
    }

    @BeforeMethod
    public void beforeMethod() throws Exception {
        JPA = new JPASetup(TM.databaseProduct, persistenceUnitName, hbmResources);
        // Zawsze usuwaj schemat w celu czyszczenia co najmniej niektórych artefaktów,
        // które mogły pozostać po ostatnim uruchomieniu w przypadku, gdyby właściwie nie wykonano
        // operacji czyszczenia
        JPA.dropSchema();

        JPA.createSchema();
        afterJPABootstrap();
    }

    public void afterJPABootstrap() throws Exception {
    }

    @AfterMethod(alwaysRun = true)
    public void afterMethod() throws Exception {
        if (JPA != null) {
            beforeJPAClose();
            if (!"true".equals(System.getProperty("keepSchema"))) {
                JPA.dropSchema();
            }
            JPA.getEntityManagerFactory().close();
        }
    }

    public void beforeJPAClose() throws Exception {

    }

    protected long copy(Reader input, Writer output) throws IOException {
        char[] buffer = new char[4096];
        long count = 0;
        int n;
        while (-1 != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
            count += n;
        }
        return count;
    }

    protected String getTextResourceAsString(String resource) throws IOException {
        InputStream is = this.getClass().getClassLoader().getResourceAsStream(resource);
        if (is == null) {
            throw new IllegalArgumentException("Nie znaleziono zasobu: " + resource);
        }
        StringWriter sw = new StringWriter();
        copy(new InputStreamReader(is), sw);
        return sw.toString();
    }

    protected Throwable unwrapRootCause(Throwable throwable) {
        return unwrapCauseOfType(throwable, null);
    }

    protected Throwable unwrapCauseOfType(Throwable throwable, Class<? extends Throwable> type) {
        for (Throwable current = throwable; current != null; current = current.getCause()) {
            if (type != null && type.isPrzypisz wartość PKableFrom(current.getClass()))
                return current;
            throwable = current;
        }
        return throwable;
    }
}

