package org.jpwh.web.dao;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Path;
import javax.persistence.metamodel.Bindable;
import javax.persistence.metamodel.SingularAttribute;
import java.util.Arrays;

public abstract class Page {

    public static enum SortDirection {
        ASC,
        DESC
    }

    /* 
         W modelu jest przechowywany rozmiar każdej strony oraz liczba rekordów wyświetlanych 
		 na każdej stronie. Wartość –1 ma specjalny znaczenie — "brak ograniczeń; wyświetl wszystkie rekordy".
     */
    protected int size = -1;

    /* 
       Utrzymywanie łącznej liczby rekordów jest konieczne do wykonania pewnych obliczeń: 
	   na przykład, aby określić, czy istnieje "następna" strona.
     */
    protected long totalRecords;

    /* 
        Stronicowanie zawsze wymaga deterministiycznej kolejności rekordów. Sortowanie 
		zazwyczaj odbywa się według konkretnego atrybutu klas encji w porządku rosnącym 
		lub malejącym. javax.persistence.metamodel.SingularAttribute 
		to w JPA atrybut encji, albo osadzalnej klasy. 
		To nie może być kolekcja (w zapytaniu nie można "porządkować według kolekcji").
     */
    protected SingularAttribute sortAttribute;
    protected SortDirection sortDirection;

    /* 
		
		Lista allowedAttributes jest ustawiana podczas tworzenia modelu strony. 
		Ogranicza ona możliwe sortowalne atrybuty do tych, które można obsługiwać w zapytaniach.
     */
    protected SingularAttribute[] allowedAttributes;

    protected Page(int size,
                   long totalRecords,
                   SingularAttribute defaultAttribute,
                   SortDirection defaultDirection,
                   SingularAttribute... allowedAttributes) {
        this.size = size;
        this.totalRecords = totalRecords;
        this.sortDirection = defaultDirection;
        this.allowedAttributes = allowedAttributes;
        setSortAttribute(defaultAttribute);
    }

    public int getSize() {
        return size;
    }

    public void setSize(int size) {
        this.size = size;
    }

    public long getTotalRecords() {
        return totalRecords;
    }

    public void setTotalRecords(long totalRecords) {
        this.totalRecords = totalRecords;
    }

    public SingularAttribute[] getAllowedAttributes() {
        return allowedAttributes;
    }

    public SingularAttribute getSortAttribute() {
        return sortAttribute;
    }

    public SortDirection getSortDirection() {
        return sortDirection;
    }

    public void setSortDirection(SortDirection sortDirection) {
        this.sortDirection = sortDirection;
    }

    public boolean isSortedAscending() {
        return SortDirection.ASC.equals(getSortDirection());
    }

    public void setAllowedAttributes(SingularAttribute[] allowedAttributes) {
        this.allowedAttributes = allowedAttributes;
    }

    public void setSortAttribute(SingularAttribute attribute) {
        if (attribute == null)
            return;
        if (!Arrays.asList(allowedAttributes).contains(attribute)) {
            throw new IllegalArgumentException(
                "Sortowanie według atrybutu: " + attribute.getName() + " jest niedozwolone"
            );
        }
        this.sortAttribute = attribute;
    }

    public boolean isMoreThanOneAvailable() {
        return getTotalRecords() != 0 && getTotalRecords() > getSize();
    }

    public boolean isAttributeDeclaredIn(SingularAttribute attribute, Bindable bindable) {
        return attribute != null && attribute.getDeclaringType().equals(bindable);
    }

    public boolean isApplicableFor(Bindable bindable) {
        return isAttributeDeclaredIn(getSortAttribute(), bindable);
    }

    public void throwIfNotApplicableFor(Path attributePath) {
        if (!isApplicableFor(attributePath.getModel())) {
            throw new IllegalArgumentException(
                "Ustawienia stronicowania/atrybut sortowania nie zostały zadeklarowane " +
                    "przez model ścieżki zapytania:" + attributePath
            );
        }
    }

    abstract public <T> TypedQuery<T> createQuery(
        EntityManager em,
        CriteriaQuery<T> criteriaQuery,
        Path attributePath
    );
}
