package org.jpwh.env;

import org.hibernate.internal.util.StringHelper;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import java.util.HashMap;
import java.util.Map;

/**
 * Tworzy obiekt EntityManagerFactory.
 * <p>
 * Konfiguracja jednostek utrwalania jest pobierana z pliku <code>META-INF/persistence.xml</code>
 * i innych źródeł. Do konstruktora można przekazać dodatkowe nazwy plików <code>hbm.xml</code> 
 * </p>
 */
public class JPASetup {

    protected final String persistenceUnitName;
    protected final Map<String, String> properties = new HashMap<>();
    protected final EntityManagerFactory entityManagerFactory;

    public JPASetup(DatabaseProduct databaseProduct,
                    String persistenceUnitName,
                    String... hbmResources) throws Exception {

        this.persistenceUnitName = persistenceUnitName;

        // Hibernate nie wykonuje automatycznego skanowania. We wszystkich jednostkach utrwalania klasy/pakiety są wymienione jawnie
        properties.put(
            "hibernate.archive.autodetection",
            "none"
        );

        // Naprawdę jedyny sposób, w jaki można przekazać pliki hbm.xml do jawnej jednostki 
        // utrwalania (gdzie skanowanie Hibernate jest wyłączone)
        properties.put(
            "hibernate.hbmxml.files",
            StringHelper.join(",", hbmResources != null ? hbmResources : new String[0])
        );

        // Nie chcemy powtarzać tych ustawień dla wszystkich jednostek w pliku persistence.xml, dlatego
        // tutaj ustawiono je programowo
        properties.put(
            "hibernate.format_sql",
            "true"
        );
        properties.put(
            "hibernate.use_sql_comments",
            "true"
        );

        // Wybranie dialektu SQL
        properties.put(
            "hibernate.dialect",
            databaseProduct.hibernateDialect
        );

        entityManagerFactory =
            Persistence.createEntityManagerFactory(getPersistenceUnitName(), properties);
    }

    public String getPersistenceUnitName() {
        return persistenceUnitName;
    }

    public EntityManagerFactory getEntityManagerFactory() {
        return entityManagerFactory;
    }

    public EntityManager createEntityManager() {
        return getEntityManagerFactory().createEntityManager();
    }

    public void createSchema() {
        generateSchema("create");
    }

    public void dropSchema() {
        generateSchema("drop");
    }

    public void generateSchema(String action) {
        // Pobierz istniejące właściwości EMF, przesłoń ustawienia generowania schematu dla kopii
        Map<String, String> createSchemaProperties = new HashMap<>(properties);
        createSchemaProperties.put(
            "javax.persistence.schema-generation.database.action",
            action
        );
        Persistence.generateSchema(getPersistenceUnitName(), createSchemaProperties);
    }
}
