package org.jpwh.web.dao;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Path;
import javax.persistence.metamodel.SingularAttribute;
import java.math.BigDecimal;

public class OffsetPage extends Page {

    /* 
        Dla potrzeb stronicowania bazującego na przesunięciach musimy wiedzieć, na której stronie jesteśmy. Domyślnie rozpoczynamy od strony 1.
     */

    protected int current = 1;

    public OffsetPage(int size, 
                      long totalRecords,
                      SingularAttribute defaultAttribute,
                      SortDirection defaultDirection, 
                      SingularAttribute... allowedAttributes) {
        super(size, totalRecords, defaultAttribute, defaultDirection, allowedAttributes);
    }

    public int getCurrent() {
        return current;
    }

    public void setCurrent(int current) {
        this.current = current;
    }

    public int getNext() {
        return getCurrent() + 1;
    }

    public int getPrevious() {
        return getCurrent() - 1;
    }

    public int getFirst() {
        return 1;
    }

    public long getLast() {
        long lastPage = (getTotalRecords() / getSize());
        if (getTotalRecords() % getSize() == 0)
            lastPage--;
        return lastPage + 1;
    }

    public long getRangeStart() {
        return (getCurrent() - 1) * getSize();
    }

    public int getRangeStartInteger() throws ArithmeticException {
        return new BigDecimal(getRangeStart()).intValueExact();
    }

    public long getRangeEnd() {
        long firstIndex = getRangeStart();
        long pageIndex = getSize() - 1;
        long lastIndex = Math.max(0, getTotalRecords() - 1);
        return Math.min(firstIndex + pageIndex, lastIndex);
    }

    public int getRangeEndInteger() throws ArithmeticException {
        return new BigDecimal(getRangeEnd()).intValueExact();
    }

    public boolean isPreviousAvailable() {
        return getRangeStart() + 1 > getSize();
    }

    public boolean isNextAvailable() {
        return getTotalRecords() - 1 > getRangeEnd();
    }

    @Override
    public <T> TypedQuery<T> createQuery(EntityManager em,
                                         CriteriaQuery<T> criteriaQuery,
                                         Path attributePath) {

        /* 
            Sprawdzamy, czy atrybut sortowania dla tej strony ma sens 
			dla określonej ścieżki atrybutów, a tym samym modelu używanego 
			przez zapytanie. Jeśli atrybut sortowania strony nie był dostępny 
			wewnątrz klasy modelu wymienionej w zapytaniu, 
			to metoda zgłasza wyjątek. 
			Jest to mechanizm bezpieczeństwa, który generuje opisowy komunikat 
			o błędzie w przypadku, gdy stworzymy parę nieodpowiednich ustawień 
			stronicowania z niewłaściwym zapytaniem.
			
         */
        throwIfNotApplicableFor(attributePath);

        CriteriaBuilder cb = em.getCriteriaBuilder();

        /* 
            Dodanie do kwerendy klauzuli <code>ORDER BY</code>.
         */
        Path sortPath = attributePath.get(getSortAttribute());
        criteriaQuery.orderBy(
            isSortedAscending() ? cb.asc(sortPath) : cb.desc(sortPath)
        );

        TypedQuery<T> query = em.createQuery(criteriaQuery);

        /* 
            Ustawienie przesunięcia zapytania. Początkowy wiersz wyniku.
         */
        query.setFirstResult(getRangeStartInteger());

        /* 
            Obcięcie wyniku z wykorzystaniem pożądanego rozmiaru strony.
         */
        if (getSize() != -1)
            query.setMaxResults(getSize());

        return query;
    }
}