package org.jpwh.test.stateless;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.persistence.UsingDataSet;
import org.jboss.arquillian.testng.Arquillian;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.jboss.shrinkwrap.resolver.api.maven.Maven;
import org.jpwh.model.Bid;
import org.jpwh.model.InvalidBidException;
import org.jpwh.model.Item;
import org.jpwh.stateless.AuctionService;
import org.testng.annotations.Test;

import javax.ejb.EJBException;
import javax.inject.Inject;
import javax.persistence.OptimisticLockException;
import java.math.BigDecimal;
import java.util.List;

import static org.testng.Assert.*;

@UsingDataSet("testdata.xml")
public class AuctionServiceTest extends Arquillian {

    @Deployment
    public static WebArchive createTestArchive() {
        WebArchive archive = ShrinkWrap.create(WebArchive.class)
            .addPackages(true, "org.jpwh.stateless") // Rekurencyjne dodanie wszystkich klas w tym pakiecie
            .addAsLibraries(Maven.resolver() // Umieść zależności Maven w WEB-INF/lib
                .loadPomFromFile("pom.xml")
                .importRuntimeDependencies()
                .resolve()
                .withTransitivity()
                .asFile());

        System.out.println("=> Instalacja wirtualnego pakietu WAR testów integracyjnych:");
        System.out.println(archive.toString(true));
        return archive;
    }

    @Inject
    AuctionService service;

    @Test
    public void editItemName() throws Exception {
        // Klient będzie musiał zarządzać stanem aplikacji. Lista przedmiotów
        List<Item> items;

        // Pobranie wszystkich przedmiotów w stanie odłączonym
        items = service.getItems(true); // Załadowanie również ofert
        assertEquals(items.size(), 5);

        // Pobranie pierwszego przedmiotu i zmiana nazwy
        Item detachedItem = items.get(0);
        assertEquals(detachedItem.getName(), "Rękawica do baseballa");
        detachedItem.setName("Piękna rękawica do baseballa");

        // Wywołanie usługi i utrwalenie zmiany. Zwraca bieżący egzemplarz Item
        detachedItem = service.storeItem(detachedItem);

        // Stary egzemplarz jest nieaktualny. Jeśli ktoś będzie się nim posługiwał, nie uda się mu go zapisać
        Item oldItem = items.get(0);
        oldItem.setName("Inna nazwa");
        boolean test = false;
        try {
            service.storeItem(oldItem);
        } catch (EJBException ex) {
            if (ex.getCause() instanceof OptimisticLockException)
                test = true;
        }
        assertTrue(test, "Powinien być zgłoszony wyjątek OptimisticLockException");

        List<Item> testItems = service.getItems(false);
        assertEquals(testItems.get(0).getName(), "Piękna rękawica do baseballa");
    }

    @Test
    public void placeBidFailure() throws Exception {
        // Klient będzie musiał zarządzać stanem aplikacji. Lista przedmiotów
        List<Item> items;

        // Pobranie wszystkich przedmiotów w stanie odłączonym
        items = service.getItems(true); // Załadowanie również ofert

        // Pobranie pierwszego przedmiotu
        Item item = items.get(0);

        // Sprawdzenie bieżącej liczby ofert i zgłoszenie nowej, wyższej oferty
        assertEquals(item.getBids().size(), 3);
        BigDecimal highestAmount = item.getHighestBid().getAmount();
        Bid newBid = new Bid(new BigDecimal(highestAmount.intValue() + 1), item);

        // Naiwny klient mógłby próbować wykonać operację, która nie jest wspierana. Dokumentujemy ją!
        item.getBids().add(newBid);

        // "Być może ta usługa realizuje kaskadowe utrwalanie za pośrednictwem adnotacji Item#bids @OneToMany?"
        item = service.storeItem(item);
        // ... nic się nie dzieje, oferta nie będzie zapisana!

        List<Item> testItems = service.getItems(true);
        assertEquals(testItems.get(0).getBids().size(), 3);
        assertEquals(testItems.get(0).getHighestBid().getAmount(), highestAmount); // Nadal stara wartość
    }

    @Test
    public void placeBid() throws Exception {
        // Klient będzie musiał zarządzać stanem aplikacji. Lista przedmiotów
        List<Item> items;

        // Pobranie wszystkich przedmiotów w stanie odłączonym
        items = service.getItems(true); // Załadowanie również ofert

        // Pobranie pierwszego przedmiotu
        Item item = items.get(0);

        // Sprawdzenie bieżącej liczby ofert i zgłoszenie nowej, wyższej oferty
        assertEquals(item.getBids().size(), 3);
        BigDecimal highestAmount = item.getHighestBid().getAmount();
        Bid newBid = new Bid(new BigDecimal(highestAmount.intValue() + 1), item);

        // Wywołanie usługi i złożenie oferty. Zwraca bieżący egzemplarz Item
        item = service.placeBid(newBid);

        // Stary egzemplarz jest nieaktualny. Jeśli ktoś będzie się nim posługiwał, nie uda się mu złożyć oferty
        Item oldItem = items.get(0);
        newBid = new Bid(new BigDecimal(newBid.getAmount().intValue() + 1), oldItem);
        boolean test = false;
        try {
            service.placeBid(newBid);
        } catch (EJBException ex) {
            if (ex.getCause() instanceof OptimisticLockException)
                test = true;
        }
        assertTrue(test, "Powinien być zgłoszony wyjątek OptimisticLockException");

        newBid = new Bid(newBid.getAmount().subtract(new BigDecimal(1)), item); // Kwota jest zbyt niska
        test = false;
        try {
            service.placeBid(newBid);
        } catch (InvalidBidException ex) {
            test = true;
        }
        assertTrue(test, "Powinien być zgłoszony wyjątek InvalidBidException");

        List<Item> testItems = service.getItems(true);
        assertEquals(testItems.get(0).getBids().size(), 4);
    }

}

