package fileLock;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.nio.file.*;
import java.security.*;
import java.security.spec.*;
import java.util.*;

import javax.crypto.*;
import javax.crypto.spec.*;

/**
 * Ten program zapisuje nazwy użytkowników i hasła do pliku passwords.properties.
 * Plik jest aktualizowany w sposób atomowy. W systemie Linux spróbuj wykonać polecenie:
 *    for f in $(seq 1 100) ; do (java fileLock.FileLockDemo user$f secret$f &) ; done
 * Po zakończeniu wszystkich procesów Javy plik passwords.properties będzie zawierał wpis 
 * dla każdego z nich.
 * 
 * @version 1.0 2023-10-27
 * @author Cay Horstmann
 */


public class FileLockDemo
{
   public static void main(String[] args)
         throws IOException, NoSuchAlgorithmException, InvalidKeySpecException
   {
      Path path = Path.of("password.properties");
      FileChannel channel = FileChannel.open(path, StandardOpenOption.READ,
            StandardOpenOption.WRITE, StandardOpenOption.CREATE);
      try (FileLock lock = channel.lock())
      {
         // Odczyt bajtów z bufora
         ByteBuffer readBuffer = ByteBuffer.allocate((int) channel.size());
         channel.read(readBuffer);
         
         // Odczyt właściwości z bajtów
         Reader in = new StringReader(new String(readBuffer.array()));
         Properties props = new Properties();
         props.load(in);
         
         // Pobranie nazwy użytkownika i hasła z argumentów 
         // lub danych wprowadzonych przez użytkownika
         String username;
         char[] password;
         if (args.length >= 2)
         {
            username = args[0];
            password = args[1].toCharArray();
         }
         else
         {
            Console console = System.console();
            username = console.readLine("Nazwa użytkownika: ");
            password = console.readPassword("Hasło: ");
         }
         
         // Wyznaczenie skrótu hasła
         final int ITERATIONS = 210_000;
         final int SALT_BYTES = 16;
         final int HASH_BYTES = 32;
         SecureRandom random = new SecureRandom();
         byte[] salt = new byte[SALT_BYTES];
         random.nextBytes(salt);

         var spec = new PBEKeySpec(password, salt, ITERATIONS, 8 * HASH_BYTES);
         SecretKeyFactory skf = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA256");
         byte[] hash = skf.generateSecret(spec).getEncoded();
         Base64.Encoder encoder = Base64.getEncoder();
         
         // Zapisanie nazwy użytkownika i skrótu hasła we właściwości
         props.put(username, ITERATIONS + "|" + encoder.encodeToString(salt) + "|"
               + encoder.encodeToString(hash));

         // Zapis właściwości w postaci bajtów
         StringWriter out = new StringWriter();
         props.store(out, null);
         byte[] outBytes = out.toString().getBytes();
         
         // Zapis bajtów do pliku
         channel.truncate(0);
         ByteBuffer writeBuffer = ByteBuffer.wrap(outBytes);
         channel.write(writeBuffer);
      }
   }
}
