package interruptible;

import java.util.*;
import java.net.*;
import java.io.*;
import java.nio.channels.*;

/**
 * Program prezentujący sposób przerywania działania kanału gniazda.
 * @author Cay Horstmann
 * @version 1.1 2023-08-16
 */
public class InterruptibleSocketTest
{
   public static void main(String[] args)
   {      
      Thread.ofPlatform().start(new TestServer());
      Scanner console = new Scanner(System.in);
      System.out.print("Użyć gniazda (S) czy kanału (C)? Naciśnij S lub C: ");
      boolean socket = console.nextLine().equalsIgnoreCase("S");
      System.out.print("Wątek systemowy (P) czy wirtualny (V)? Naciśnij P lub V: ");
      boolean platform = console.nextLine().equalsIgnoreCase("P");
      Runnable client = socket ? InterruptibleSocketTest::useSocket
            : InterruptibleSocketTest::useChannel;
      Thread clientThread = platform ? Thread.ofPlatform().start(client) 
            : Thread.ofVirtual().start(client);
         
      System.out.print("Naciśnij Enter, by przerwać działanie klienta");
      console.nextLine();
      clientThread.interrupt();
   }

   /**
    * Nawiązanie połączenia z serwerem testowym przy użyciu kanału
    */
   public static void useChannel()
   {
      System.out.println("Kanał:\n");
      try (SocketChannel channel 
            = SocketChannel.open(new InetSocketAddress("localhost", 8189)))
      {
         Scanner in = new Scanner(channel);
         while (!Thread.currentThread().isInterrupted())
         {
            System.out.print("Odczytuję ");
            if (in.hasNextLine())
            {
               String line = in.nextLine();
               System.out.println(line);
            }
         }
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }
      finally
      {
         System.out.println("Kanał zamknięty\n");
      }
   }

   /**
    * Nawiązanie połączenia z serwerem testowym przy użyciu gniazda
    */
   public static void useSocket()
   {
      System.out.println("Gniazdo:\n");
      try (var sock = new Socket("localhost", 8189))
      {
         Scanner in = new Scanner(sock.getInputStream());
         while (!Thread.currentThread().isInterrupted())
         {
            System.out.print("Odczytuję ");
            if (in.hasNextLine())
            {
               String line = in.nextLine();
               System.out.println(line);
            }
         }
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }
      finally
      {
         System.out.println("Gniazdo zamknięte\n");
      }
   }

   /**
    * Serwer nasłuchujący na porcie 8189, który wysyła liczby do klienta
    * i symuluje zawieszenie się po wysłaniu 10 liczb.
    * 
    */
   static class TestServer implements Runnable
   {
      public void run()
      {
         try (var s = new ServerSocket(8189);
            Socket incoming = s.accept())
         {
            serve(incoming);
         }
         catch (Exception e)
         {
            e.printStackTrace();;
         }
         finally
         {
            System.out.println("Zamykanie połączenia\n");
         }
      }

      private void serve(Socket incoming) throws IOException, InterruptedException
      {
         int counter = 0;
         try (var out = new PrintWriter(incoming.getOutputStream(),
               true /* autoFlush */))
         {
            while (counter < 100)
            {
               counter++;
               if (counter <= 10) out.println(counter);
               Thread.sleep(100);
            }
         }
      }
   }
}

