package com.oreilly.servlet;

import java.io.*;
import java.util.*;

import com.oreilly.servlet.LocaleToCharsetMap;

public class LocaleNegotiator {

  private ResourceBundle chosenBundle; 
  private Locale chosenLocale; 
  private String chosenCharset; 

  public LocaleNegotiator(String bundleName,
                          String languages,
                          String charsets) {

    // Okrelenie wartoci domylnych:
    //   Jzyk angielski, kodowanie ISO-8859-1 (Latin-1), pakiet angielski
    Locale defaultLocale = new Locale("en", "US");
    String defaultCharset = "ISO-8859-1";
    ResourceBundle defaultBundle = null;
    try {
      defaultBundle = ResourceBundle.getBundle(bundleName, defaultLocale);
    }
    catch (MissingResourceException e) {
      // Nie odnaleziono domylnego pakietu. Dziaanie bez zabezpieczenia.
    }

    // Jeeli klient nie okreli przyjmowanych jzykw, mona zatrzyma domylne
    if (languages == null) {
      chosenLocale = defaultLocale;
      chosenCharset = defaultCharset;
      chosenBundle = defaultBundle;
      return;  // szybkie wyjcie
    }

    //Wykorzystanie znacznikw do oddzielenia akceptowanych jzykw
    StringTokenizer tokenizer = new StringTokenizer(languages, ",");

    while (tokenizer.hasMoreTokens()) {
      // Pobranie nastpnego przyjmowanego jzyka
      // (Jzyk moe wyglda tak: "en; wartoscq=0.91")
      String lang = tokenizer.nextToken();

      // Pobranie lokalizacji dla danego jzyka
      Locale loc = getLocaleForLanguage(lang);

      // Pobranie pakietu dla tej lokalizacji. Nie naley pozwoli na dopasowanie innych 
      // jzykw!
      ResourceBundle bundle = getBundleNoFallback(bundleName, loc);

      // Zwrcony pakiet wynosi null, jeeli nie znaleziono doapsowania. W tym przypadku 
      // nie mona wykorzysta tego jzyka, poniewa serwlet go nie zna.
      if (bundle == null) continue;  // przejcie do nastpnego jzyka

      // Odnalezienie kodowania, ktre mona wykorzysta do wywietlenia jzyka tej 
      // lokalizacji
      String charset = getCharsetForLocale(loc, charsets);

      // Zwrcone kodowanie wynosi null, jeeli nie znaleziono doapasowania. W tym 
      // przypadku nie mona wykorzysta tego jzyka, poniewa serwlet nie moe go 
      // zakodowa
      if (charset == null) continue;  // on to the next language

      // Jeeli w tym miejscu, to nie ma problemw z tym jzykiem.
      chosenLocale = loc;
      chosenBundle = bundle;
      chosenCharset = charset;
      return;  // koniec
    }

    // Brak dopasowania, zostaje domylny
    chosenLocale = defaultLocale;
    chosenCharset = defaultCharset;
    chosenBundle = defaultBundle;
  }

  public ResourceBundle getBundle() {
    return chosenBundle;
  }

  public Locale getLocale() {
    return chosenLocale;
  }

  public String getCharset() {
    return chosenCharset;
  }

  private Locale getLocaleForLanguage(String lang) {
    Locale loc;
    int semi, dash;

    // Odcicie wszystkich wartoci q, ktre mog nastpowa po redniku
    if ((semi = lang.indexOf(';')) != -1) {
      lang = lang.substring(0, semi);
    }

    // Obcicie wolnych miejsc
    lang = lang.trim();

    // Utworzenie Locale z jzyka. Mylnik moe oddziela jzyk od kraju
    if ((dash = lang.indexOf('-')) == -1) {
      loc = new Locale(lang, "");  // Brak mylnika, brak kraju
    }
    else {
      loc = new Locale(lang.substring(0, dash), lang.substring(dash+1));
    }

    return loc;
  }

  private ResourceBundle getBundleNoFallback(String bundleName, Locale loc) {

    // Po pierwsze pobranie pakietu awaryjnego  pakietu, ktry zostanie pobrany, jeeli
    // getBundle() nie moe odnale bezporedniego doapasowania. Pakiet ten bdzie 
    // porwnywany z pakietami dostarczanymi przez nastpne wywoania getBundle() w celu 
    // wykrycia, czy getBundle() znalaza bezporednie dopasowanie.
    ResourceBundle fallback = null;
    try {
      fallback =
        ResourceBundle.getBundle(bundleName, new Locale("bogus", ""));
    }
    catch (MissingResourceException e) {
      // Nie odnaleziono pakietu awaryjnego
    }

    try {
      // Pobranie pakietu dla okrelonej lokalizacji
      ResourceBundle bundle = ResourceBundle.getBundle(bundleName, loc);

      // Czy pakiet rni si od pakietu awaryjnego?
      if (bundle != fallback) {
        // Prawdziwe dopasowanie!
        return bundle;
      }
      // Tak wic pakiet jest taki sam, jak pakiet awaryjny.
      // To cigle moe by dopasowanie, ale tylko wtedy, gdy lokalny jzyk pasuje do 
      // jzyka domylnej lokalizacji.
      else if (bundle == fallback &&
            loc.getLanguage().equals(Locale.getDefault().getLanguage())) {
        // Inny sposb na dopasowanie
        return bundle;
      }
      else {
        // Brak dopasowania, kontynuacja poszukiwa
      }
    }
    catch (MissingResourceException e) {
      // Brak pakietu dla tej lokalizacji
    }

    return null;  // Brak dopasowania
  }

  protected String getCharsetForLocale(Locale loc, String charsets) {
    // Uwaga  ta metoda ignoruje kodowania okrelone przez klienta
    return LocaleToCharsetMap.getCharset(loc);
  }
}
