package sched;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.StringTokenizer;

import javax.validation.constraints.NotNull;

/** 
 * Prosta klasa spotkania (może być cykliczne).
 * Klasa jest częściowo niezmienna - jej główne pola są niezmienne, natomiast
 * informacje o powtarzaniu spotkania można zmieniać.
 * @author Ian Darwin
 */
public class Appt implements Comparable<Appt> {

    //-----------------------------------------------------------------
    //    ZMIENNE KLASY GŁÓWNEJ -- SPOTKANIE
    //-----------------------------------------------------------------
    /** Co musimy zrobić tym razem. */
    @NotNull
    final String text;    
    /** Dzień w którym odbędzie się spotkanie. */
    @NotNull
    final LocalDate date;
    /** Godzina - opcjonalna dla elementów typu TODO (do zrobienia). */
    final LocalTime time;
    //-----------------------------------------------------------------
    //    STAŁE NA POTRZEBY POWTARZANIA SPOTKAŃ.
    //-----------------------------------------------------------------
    /** Typ oznaczający brak powtórzeń. */
    public static final int NONE = 0;
    /** Typ oznaczający brak powtórzeń. */
    public static final int HOURLY = 1;
    /** Typ dla powtórzeń co godzinę. */
    public static final int DAILY = 2;
    /** Typ dla powtórzeń co tydzień. */
    public static final int WEEKLY = 3;
    /** Powtarzanie co miesiąc o znaczeniu "12. dzień każdego miesiąca". */
    public static final int MONTHLY_NUMDAY_OF_M = 41;
    /** Powtarzanie co miesiąc o znaczeniu "2. czwartek każdego miesiąca". */
    public static final int MONTHLY_WEEKDAY_OF_M = 42;
    /** Typ dla powtarzania co roku. */
    public static final int YEARLY = 5;
    /** Czynnik ilości oznaczający "w nieskończoność". */
    public static final int FOREVER = Integer.MAX_VALUE;

    //-----------------------------------------------------------------
    //    ZMIENNE KLASY GŁÓWNEJ -- POWTÓRZENIA
    //-----------------------------------------------------------------
    /** Typo powtórzeń dla tego powtarzanego obiektu. */
    protected int r_type = NONE;
    /** Przedział powtórzeń: 2=każdy następny (godzina, dzień, miesiąc, rok) */
    protected int r_interval = NONE;
    /** Liczba powtórzeń tego wydarzenia */
    protected int r_count = NONE;

    //-----------------------------------------------------------------
    //    METODY - KONSTRUKTORY
    //-----------------------------------------------------------------
    /** Konstruktor. */
    public Appt(int y, int mo, int d, int h, int min, String text) {
        this.text = text;
        date = LocalDate.of(y, mo, d);
        time = LocalTime.of(h, min);
    }
    
    public Appt(LocalDateTime datetime, String text) {
        this.text = text;
        this.date = datetime.toLocalDate();
        this.time = datetime.toLocalTime();
    }
    
    public Appt(LocalDate date, LocalTime time, String text) {
        this.text = text;
        this.date = date;
        this.time = time;
    }

    //-----------------------------------------------------------------
    //    METODY - POWTÓRZENIA
    //-----------------------------------------------------------------
    public void setRep(int typ, int intv, int count) {
        r_type = typ;
        r_interval = intv;
        r_count = count;
    }

    /** Metoda określaczy dane spodtanie odpowiada dacie podanej jako:
     *  y - rok, m - miesiąc, d - dzień.
     */
    public boolean matches(int y, int m, int d) {
        // Na początku sprawdzamy prosty przypadek.
        LocalDate dt = LocalDate.of(y, m, d);
        if (date.equals(dt))
            return true;
        // Jeśli to NIE jest dziś I nie powtarza się, to to spotkanie nas nie interesuje.
        if (r_count == NONE)
            return false;

        // Potenajalnie jesteśmy zainteresowani!
        LocalDate newDay = date;

        for (int i=0; i<r_count && date.compareTo(dt) < 0; i++) {
            switch(r_type) {
            case HOURLY:
                break;
            case DAILY:
                newDay = date.plusDays(r_interval);
                break;
            case WEEKLY:
                break;
            case MONTHLY_NUMDAY_OF_M:
                break;
            case MONTHLY_WEEKDAY_OF_M:
                break;
            case YEARLY:
                break;
            }

            // OK, inkrementacja wykonana. Teraz spradzimy czy data
            // pasuje do tej, której poszukujemy.
            if (newDay.equals(date))
                return true;
        }

        // Pętla zakończona bez znalezienia poszukiwanej daty, zatem...
        return false;
    }

    // tag::main[]
// public class Appt implements Comparable {
    // Klasa nie jest kompletna - patrz przykłady dołączone do książki.
    //-----------------------------------------------------------------
    //    METODY - PORÓWNYWANIE
    //-----------------------------------------------------------------
    /** compareTo metoda, interfejs Comparable.
     * Porównanie tego obiektu Appt z innym w celu 
     * ich odpowiedniego posortowania.
     * <P>Przy sortowaniu uwzględniane są tylko tekst, data
     * i czas, bez powtórzeń! 
     * Powtórzenia dotyczą zdarzeń powtarzających się co 
     * jakiś czas, np. spotkanie jest w każdy piątek o 9:00.
     * Zgodne z działaniem metody equals().
     * @return -1, jeśli this<a2, +1, jeśli this>a2, 0 
     * w pozostałych przypadkach.
     */
    @Override
    public int compareTo(Appt a2) {
        // Jeśli daty nie są takie same, to je porównujemy.
        int dateComp = date.compareTo(a2.date);
        if (dateComp != 0)
            return dateComp;
        // Daty takie same. Jeśli godziny są różne, to je porównujemy.
        if (time != null && a2.time != null) {
            // Obie godziny różne od null.
            int timeComp = time.compareTo(a2.time);
            if (timeComp != 0)
                return timeComp;
        } else /* Co najmniej jedna godzina. */ {
            if (time == null && a2.time != null) {
                return -1; // Spotkania całodniowe są sortowane jako mniejsze, 
                           // by były wyświetlane w pierwszej kolejności.
            } else if (time != null && a2.time == null)
                return +1;
                // Oba spotkania nie mają ustawionej godziny, idziemy dalej.
        }
        // Ta sama data i godzina, uwzględniamy tekst.
        return text.compareTo(a2.text);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((date == null) ? 0 : date.hashCode());
        result = prime * result + ((text == null) ? 0 : text.hashCode());
        result = prime * result + ((time == null) ? 0 : time.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object o2) {
        if (this == o2)
            return true;
        if (o2.getClass() != Appt.class)
            return false;
        Appt a2 = (Appt) o2;
        if (!date.equals(a2.date))
            return false;
        if (time != null && !time.equals(a2.time))
            return false;
        return text.equals(a2.text);
    }

    /** Metoda zwraca tekstową reprezentację tego obiektu Appt.
     * Jej wyniki są przeznaczone do debugowania, a nie prezentacji!
     */
    @Override
    public String toString() {
        var sb = new StringBuilder();
        sb.append(date).append(' ');
        if (time != null) {
            sb.append(time.getHour())
            .append(':')
            .append(time.getMinute())
            .append(' ');
        } else {
            sb.append("(All day)").append(' ');
        }
        sb.append(text).toString();
        return sb.toString();
    }
    // end::main[]

    /** Metoda wytwórcza: przekształca String na instancję Appt.
     * Do przemyślenia - rzutowanie jako statyczna metoda wywtórcza w celu uzyskania
     * niewielkich korzyści czasowych.
     */
    public static Appt fromString(String s) {
        System.out.println("Appt.fromString(): " + s);
        // Tokenize, after stripping '-' and ':'.
        StringTokenizer st = new StringTokenizer(s.replaceAll("[-:]", " "));
        if (st.countTokens() < 6) throw new
            IllegalArgumentException("Zby mało pól w łańcuchu: " + s);
        int y = Integer.parseInt(st.nextToken());
        int m = Integer.parseInt(st.nextToken());
        int d = Integer.parseInt(st.nextToken());
        int h = Integer.parseInt(st.nextToken());
        int i = Integer.parseInt(st.nextToken());
        StringBuilder sb = new StringBuilder();
        while (st.hasMoreElements()) {
            sb.append(st.nextToken());
            if (st./*wciąż*/hasMoreElements())
                sb.append(' ');
        }
        return new Appt(y, m, d, h, i, sb.toString());
    }

    public int getDay() {
        return date.getDayOfMonth();
    }

    public int getHour() {
        return time.getHour();
    }

    public int getMinute() {
        return time.getMinute();
    }

    public int getMonth() {
        return date.getMonthValue();
    }
    
    public String getText() {
        return text;
    }
    
    public int getYear() {
        return date.getYear();
    }
    
}
