package network;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.net.Socket;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;

/**
 * DaytimeBinary - program nawiązuje połączenie ze standardową 
 * usługą Time (binarną).
 * Klient 'rdate' pod inną nazwą.
 * @author Ian F. Darwin
 */
// tag::main[]
public class RDateClient {
    /** Port TCP używany przez usługę zwracającą czas w postaci binarnej. */
    public static final short TIME_PORT = 37;
    /** Liczba sekund pomiędzy podaną datą a 1.01.1970, punktem odniesienia przy wyznaczaniu
     * dat Date(long) oraz czasu Time w języku Java.
     * Uwzględnia lata przestępne (aż do 2100).
     * Odejmuje 1 dzień z roku 1900, dodaje pół dnia dla lat 1969 - 1970.
     */
    protected static final long BASE_DAYS = 
        (long)((1970-1900)*365 + (1970-1900-1)/4);

    /* Liczba sekund od początku roku 1970. */
    public static final long BASE_DIFF = (BASE_DAYS * 24 * 60 * 60);

    public static void main(String[] argv) {
        String hostName;
        if (argv.length == 0)
            hostName = "localhost";
        else
            hostName = argv[0];

        try (Socket sock = new Socket(hostName,TIME_PORT);) {
            DataInputStream is = new DataInputStream(new 
                BufferedInputStream(sock.getInputStream()));
            // Musimy odczytać 4 bajty z sieci jako liczby bez znaku.
            // Musimy wykonać to sami, gdyż nie ma metody readUnsignedInt().
            // Typ long w Javie ma wielkość 8 bajtów, jednak 
            // my używamy istniejącego protokołu daytime korzystającego 
            // z liczb całkowitych o długości 4 bajtów.
            long remoteTime = (
                ((long)(is.readUnsignedByte()) << 24) |
                ((long)(is.readUnsignedByte()) << 16) |
                ((long)(is.readUnsignedByte()) <<  8) |
                ((long)(is.readUnsignedByte()) <<  0));
            System.out.println("Czas na komputerze zdalnym: " + remoteTime);
            System.out.println("BASE_DIFF = " + BASE_DIFF);
            System.out.println("Różnica czasów == " + (remoteTime - BASE_DIFF));
            Instant time = Instant.ofEpochSecond(remoteTime - BASE_DIFF);
            LocalDateTime d = LocalDateTime.ofInstant(time, ZoneId.systemDefault());
            System.out.println("Czas na komputerze " + hostName + " to " + 
                                                               d.toString());
            System.out.println("Lokalna data/godzina = " + 
                                                         LocalDateTime.now());
        } catch (IOException e) {
            System.err.println(e);
        }
    }
}
// end::main[]
