/*
 * FUNKCYJNE PODEJCIE DO JAVY
 * Rozdzia 10. Obsuga wyjtkw funkcyjnych
 *
 * Listing 10.7. Dodawanie transformacji do Result<V, E>
 */

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

public class ResultTransformers {

    public record Result<V, E extends Throwable> (V value, E throwable, boolean isSuccess) {

        public static <V, E extends Throwable> Result<V, E> success(V value) {
            return new Result<>(value, null, true);
        }

        public static <V, E extends Throwable> Result<V, E> failure(E throwable) {
            return new Result<>(null, throwable, false);
        }

        public <R> Optional<R> mapSuccess(Function<V, R> fn) {
            return this.isSuccess ? Optional.ofNullable(this.value).map(fn) : Optional.empty();
        }

        public <R> Optional<R> mapFailure(Function<E, R> fn) {
            return this.isSuccess ? Optional.empty() : Optional.ofNullable(this.throwable).map(fn);
        }

        public <R> R map(Function<V, R> successFn, Function<E, R> failureFn) {
            return this.isSuccess ? successFn.apply(this.value) //
                                  : failureFn.apply(this.throwable);
        }
    }

    static Result<String, IOException> safeReadString(Path path) {
        try {
            return Result.success(Files.readString(path));
        }
        catch (IOException e) {
            return Result.failure(e);
        }
    }

    public static void main(String... args) {
        // OBSUGA TYLKO PRZYPADKU POWODZENIA 
        var successOnly = Stream.of(Paths.get("../jshell/try-catch.java"),
                                    Paths.get("nieprawidowy plik"),
                                    Paths.get("../jshell/files-readstring.java"))
                                .map(ResultTransformers::safeReadString)
                                .map(result -> result.mapSuccess(String::toUpperCase)) 
                                .flatMap(Optional::stream)
                                .toList();

        System.out.println("Znaleziono (successOnly): " + successOnly.size());

        // OBSUGA OBU PRZYPADKW
        var result = safeReadString(Paths.get("nieprawidowy plik")).map(
            success -> success.toUpperCase(),
            failure -> "Bd operwacji we-wy: " + failure.getMessage()
        );

        System.out.println("Wynik: " + result);
    }
}
