# Rozdział 12. Rekurencja


## Strumienie i rekurencjaa

Alternatywnym podejściem do rekurencji jest użycie nieskończonego potoku strumieniowego z wywołaniem rekurencyjnym opakowanym w funkcyjny interfejs.
Taki interfejs musi reprezentować wszystkie części algorytmu rekurencyjnego:

```java
@FunctionalInterface
public interface RecursiveCall<T> {

    // Reprezentuje wywołanie rekurencyjne
    RecursiveCall<T> apply();

    // Interfejs opakowujący musi otrzymywać informację, kiedy łańcuch wywołań zostanie
    // przez osiągnięcie swojego warunku bazowego
    default boolean isComplete() {
        return false;
    }

    // Ostateczny wynik wywołania rekurencyjnego
    default T result() {
        return null;
    }

    // Uruchamia potok strumieniowy
    default T run() {
        return Stream.iterate(this, RecursiveCall::apply)
                     .filter(RecursiveCall::isComplete)
                     .findFirst()
                     .get()
                     .result();
    }

    // Metoda pomocnicza do tworzenia wyrażenia lambda, które reprezentuje
    // osiągnięty warunek bazowy i zawiera rzeczywisty wynik
    static <T> RecursiveCall<T> done(T value) { <5>

        return new RecursiveCall<T>() {

            @Override
            public boolean isComplete() {
                return true;
            }

            @Override
            public T result() {
                return value;
            }

            @Override
            public RecursiveCall<T> apply() {
                return this;
            }
        };
    }
}
```

Wywołanie `run()` utworzy i uruchomi nieskończony potok strumieniowy.
Metoda `Stream.iterate(...)` stosuje wartośc początkową (`this`) do `UnaryOperator` (`this::apply`).

Wynik jest następnie stosowany iteracyjnie ponownie do `UnaryOperator` dopóki `isComplete() == true` i nie zostanie osiągnięta operacja terminalna `findFirst()` strumienia.

Ten interfejs funkcyjny jest iteracyjnym interfejem opakowującym dla wywołań w stylu rekurencyjnym; eliminuje `StackOverflowError` lub przepełnienie liczbowe w przypadku obliczania silni.

Ponieważ przykład obliczania silni powoduje dość szybkie przepełnienie liczbowe – największy możliwy +long+ to 9,223,372,036,854,775,807 lub 2^63-1, co mieści się w przedziela między 20! i 21! – wybraliśmy prostszy przykład: sumowanie liczb.

Poniższy kod reprezentuje wywołanie rekurencyjne, ale jako `RecursiveCall<Long>`:

```java
RecursiveCall<Long> sum(Long total, Long summand) {
    if (summand == 1) {
        return RecursiveCall.done(total);
    }

    return () -> sum(total + summand, summand - 1L);
}

var result = sum(1L, 4000L).run();
```

W porównaniu do aktualnej wersji rekurencyjnej jedyna różnica polega na tym, że warunek bazowy i wartość zwracana z `sum(...)` są opakowane w nową lambę `RecursiveCall`.

Niewidoczną różnicą między rekurencją a "czymś w rodzaju rekurencji" w przypadku strumieni jest głębokość stosu.
Ponieważ wersja rekurencyjna tworzy nową ramkę stosu dla każdego wywołania metody, strumień działa iteracyjnie, a zatem ma stałą głębokość stosu.
Może się ona różnić, jeśli zmieni się jego implementacja.
Mimo to zawsze będzie to spójne, niezależnie od wymaganych kroków rekurencyjnych.
Pozwala to na użycie rekurencyjnego podejścia do rozwiązania problemu wymagającego wielu zagnieżdżonych wywołań, które zwykle rzuciłyby błąd `StackOverflowError`.

Podejście do rekurencji oparte na strumieniach jest świetnym dowodem koncepcji do podkreślenia funkcyjnych możliwości Javy i strumieni.
Nie polecałbym go jednak jako rozwiązania ogólnego zastosowania "w świecie rzeczywistym".
Prosta pętla lub nawet potok strumieniowy bez wyspecjalizowanego typu `RecursiveCall` jest często wystarczający i sprawi, że kod będzie bardziej zwięzły i rozsądny.
