package math.primes;

import math.bigs.BigsUtil;
import math.utils.FrequencyMap;
import math.utils.MathUtil;
import math.utils.Tuple2L;

import java.awt.*;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;

public class PrimeUtil {
    private static final BigInteger ONE = BigInteger.valueOf(1);
    private static final BigInteger TWO = BigInteger.valueOf(2);

    private PrimeUtil() {
    }

    /**
     * zwraca pierwszą liczbę pierwszą większą od podanej liczby
     *
     * @param min int - liczba progowa
     * @return int - pierwsza liczba pierwsza większa od min
     */
    public static long getPrime(long min) {
        for (long j = min + 1; true; j++) {
            if (isPrime(j)) {
                return j;
            }
        }
    }

    public static BigInteger getPrime(BigInteger min) {
        for (BigInteger j = min.add(ONE); true; j = j.add(ONE)) {
            if (isPrime(j)) {
                return j;
            }
        }
    }

    public static boolean isPrime(long n) {
        for (long j = 2; Math.pow(j, 2) <= n; j++) {
            if (n % j == 0) {
                return false;
            }
        }
        return true;
    }

    /**
     * Perform the Lucas test.
     *
     * @return true if p is prime, false if p is composite
     */
    public static boolean testPrimeLucas(int p) {
        // Try integers a from 2 through p.
        for (int a = 2; a <= p; ++a) {
            if (passPart1Lucas(a, p) && passPart2Lucas(a, p))
                return true; // prime
        }
        return false; // composite
    }

    /**
     * zwraca pierwszą liczbę pierwszą większą od podanej liczby
     *
     * @param min int - liczba progowa
     * @return int - pierwsza liczba pierwsza większa od min
     */
    public static int getPrime(int min) {
        for (int j = min + 1; true; j++) {
            if (isPrime(j)) {
                return j;
            }
        }
    }

    /**
     * Test if integer a passes the second part of the test.
     *
     * @param a the value of a
     * @return true if [a^(p-1)/q]%p != 1 for all prime factors q,
     * else false
     */
    private static boolean passPart2Lucas(int a, int p) {
        int pm1 = p - 1;
        int[] q = factorize2(p - 1);
        // Loop to try each prime factor.
        for (int aQ : q) {
            int exponent = pm1 / aQ;
            int value = powerMod(a, exponent, p);
            // Report status back to the caller.
            if (value == 1)
                return false; // fail
        }
        return true; // pass
    }

    /**
     * Wykonuje mnożenie modularne
     *
     * @param a liczba q
     * @param b the liczba b
     * @param m the modulus
     * @return a*b (mod m))
     * faktycznie funkcja wykonuje mnożenie a*b = c
     * potem oblicza c%m, ale ta funkjca wykonuje to szybciej
     */
    public static int multMod(int a, int b, int m) {
        int wynik = 0;
        while (a > 0) {
            if ((a & 1) == 1) {
                wynik += b;
                wynik %= m;
            }
            b <<= 1;
            b %= m;
            a >>= 1;
        }
        return wynik;
    }

    /**
     * Wykonuje potęgowanie modularne
     *
     * @param a liczba a
     * @param b liczba b
     * @param m modulus
     * @return wynik potęgowania modularnego. Właściwie
     * funkcja podnosi a do potęgi b = c
     * oblicza c%modulus, ale wykonuje to szybciej
     */
    public static int powerMod(int a, int b, int m) {
        int wynik = 1;
        while (b > 0) {
            if ((b & 1) == 1) {
                wynik = multMod(wynik, a, m);
            }
            a = multMod(a, a, m);
            b >>= 1;
        }
        return wynik;
    }

    /**
     * Test if integer a passes the first part of the test.
     *
     * @param a the value of a
     * @return true if [a^(p-1)]%p == 1, else false
     */
    private static boolean passPart1Lucas(int a, int p) {
        int exponent = p - 1;
        int value = powerMod(a, exponent, p);
        // Report status back to the caller.
        return (value == 1); // pass if it's 1
    }

    public static boolean isPrime(BigInteger n) {
        for (BigInteger j = TWO; BigsUtil.mniejszy(j.pow(2), n)
                || BigsUtil.rowny(j.pow(2), n); j = j.add(ONE)) {
            if (n.mod(j).equals(BigInteger.ZERO)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Generuje liczbę pierwszą Eulera
     *
     * @param n - liczba wyjściowa
     * @return - liczbę pierwszą Eulera
     */
    public static long euler(long n) {
        return (long) Math.pow(n, 2L) - 79L * n + 1601L;
    }

    public static BigInteger euler(BigInteger n) {
        return n.pow(2).subtract(BigInteger.valueOf(79).multiply(n))
                .add(BigInteger.valueOf(1601));
    }

    /**
     * Oblicza liczbę Mersenna = Math.pow(2, exp)-1, ale
     * robi to szybciej
     */
    public static BigInteger generateMersenneNumber(int exp) {
        BigInteger p = ONE;
        for (int i = 1; i <= exp; ++i) {
            p = p.add(p);
        }
        p = p.subtract(ONE);
        return p;
    }

    /**
     * sprawdza czy podana liczba jest liczbą pierwszą
     *
     * @param n int - liczba do zbadania
     * @return boolean - zwraca true jeśli badana liczba jest liczbą
     * pierwszą, a false w przeciwnym wypadku
     */
    public static boolean isPrime(int n) {
        for (int j = 2; Math.pow(j, 2) <= n; j++) {
            if (n % j == 0) {
                return false;
            }
        }
        return true;
    }

    public static long generateMersenneNumber2(int exp) {
        return (long) Math.pow(2L, exp) - 1L;
    }

    /**
     * oblicza liczby pierwsze w podanym zakresie
     *
     * @param min long - pierwsza liczba
     * @param max long - większa liczba
     * @return ArrayList<Integer> - arraylista liczb pierwszych
     * leżących między min, a max
     */
    public static ArrayList<Long> getPrimes(long min, long max) {
        ArrayList<Long> list = new ArrayList<>();
        for (long j = min + 1; j < max; j++) {
            if (isPrime(j)) {
                list.add(j);
            }
        }
        return list;
    }

    public static void printBlizniacze(long min, long max, String path) {
        PrintWriter pw = null;
        try {
            pw = new PrintWriter(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        for (long j = min + 1; j < max; j++) {
            if (isPrime(j) && isPrime(j + 2)) {
                if (pw != null) {
                    pw.println("<p>(" + j + ", " + (j + 2) + ")</p>");
                }
            }
        }
        close(pw);
    }

    public static void printLustrzane(int min, int max, String path) {
        PrintWriter pw = null;
        try {
            pw = new PrintWriter(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        for (int j = min + 1; j < max; j++) {
            int a = MathUtil.reverteInt(j);
            if (isPrime(j) && isPrime(a) && (j != a)) {
                if (pw != null) {
                    pw.println("<p>(" + j + ", " + a + ")</p>");
                }
            }
        }
        close(pw);
    }

    public static void printPalindrom(int min, int max, String path) {
        PrintWriter pw = null;
        try {
            pw = new PrintWriter(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        for (int j = min + 1; j < max; j++) {
            int a = MathUtil.reverteInt(j);
            if (isPrime(j) && isPrime(a) && (j == a)) {
                if (pw != null) {
                    pw.println(
                            "<p>(" + j + ", " + MathUtil.reverteInt(j) + ")</p>");
                }
            }
        }
        close(pw);
    }

    public static void printGermain(long min, long max, String path) {
        PrintWriter pw = null;
        try {
            pw = new PrintWriter(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        for (long j = min + 1; j < max; j++) {
            if (isPrime(j) && isPrime((j * 2) + 1)) {
                if (pw != null) {
                    pw.println("<p>" + j + " (" + (j * 2 + 1) + ")</p>");
                }
            }
        }
        close(pw);
    }

    public static void printIzolowane(int min, int max, String path) {
        PrintWriter pw = null;
        try {
            pw = new PrintWriter(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        ArrayList<Integer> al = getPrimes(min, max);
        for (int i = 1; i < al.size() - 1; i++) {
            int t1 = al.get(i - 1);
            int t2 = al.get(i);
            int t3 = al.get(i + 1);
            if (((t2 - t1) >= 4) && ((t3 - t2) >= 4)) {
                if (pw != null) {
                    pw.println("<p>" + t2 + "</p>");
                }
            }
        }
        close(pw);
    }

    public static void printCzworacze(long min, long max, String path) {
        PrintWriter pw = null;
        try {
            pw = new PrintWriter(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        for (long j = min + 1; j < max; j++) {
            if (isPrime(j) && isPrime(j + 2) && isPrime(j + 6)
                    && isPrime(j + 8)) {
                if (pw != null) {
                    pw.println("<p>(" + j + ", " + (j + 2) + ", " + (j + 6) + ", "
                            + (j + 8) + ")</p>");
                }
            }
        }
        close(pw);
    }

    /**
     * Oblicza liczbę liczb pierwszych w podanym przedziale
     *
     * @param min - początek przedziału
     * @param max - koniec przedziału
     * @return - zwraca liczbę liczb pierwszych w podanym przedziale
     */
    public static long getNPrimes(long min, long max) {
        long sum = 0;
        for (long j = min + 1; j < max; j++) {
            if (isPrime(j)) {
                sum++;
            }
        }
        return sum;
    }

    public static ArrayList<BigInteger> getPrimes(BigInteger min,
                                                  BigInteger max) {
        ArrayList<BigInteger> list = new ArrayList<>();
        for (BigInteger j = min.add(ONE); BigsUtil.mniejszy(j,
                max); j = j.add(ONE)) {
            if (isPrime(j)) {
                list.add(j);
            }
        }
        return list;
    }

    /**
     * Sito Eratostenesa
     *
     * @param max - liczba maksymalna
     * @return tablicę booleanów, w której indeks oznacza liczbę
     * a wartość true w komórce - liczbę pierwsza, Gdy wartość w komórce
     * jest false - liczba nie jest pierwsza
     */
    public static boolean[] sitoEratostenesa(int max) {
        int pol = (max + 1) >> 1;
        boolean wynik[] = new boolean[max + 1];
        for (int i = 2; i <= max; ++i) {
            wynik[i] = true;
        }
        int p = 2;
        while (p < pol) {
            for (int z = p << 1; z <= max; z += p) {
                wynik[z] = false;
            }
            //while((++p < pol) && (!wynik[p]));
        }
        return wynik;
    }

    public static boolean[] sitoAtkina(int max) {
        boolean[] primes = new boolean[max + 1];
        primes[2] = true;
        primes[3] = true;
        int root = (int) Math.ceil(Math.sqrt(max));
        for (int x = 1; x < root; x++) {
            for (int y = 1; y < root; y++) {
                int n = 4 * x * x + y * y;
                if (n <= max && (n % 12 == 1 || n % 12 == 5))
                    primes[n] = !primes[n];
                n = 3 * x * x + y * y;
                if (n <= max && n % 12 == 7)
                    primes[n] = !primes[n];
                n = 3 * x * x - y * y;
                if ((x > y) && (n <= max) && (n % 12 == 11))
                    primes[n] = !primes[n];
            }
        }
        for (int i = 5; i <= root; i++)
            if (primes[i])
                for (int j = i * i; j < max; j += i * i)
                    primes[j] = false;
        return primes;
    }

    /**
     * Test pierwszości liczby
     *
     * @param p     - liczba do przetestowania
     * @param iters - liczba iteracji - najlepiej około 1000;
     * @return false jeśli liczba nie jest pierwsza i jest
     * to całkowicie pewne, true, jeśli liczba jest pierwsza
     * (z bardzo dużym prawdopodobieństwem zależnym od liczby iteracji -
     * im więcej iteracji tym większe prawdopodobieństwo, że liczba jest
     * pierwsza
     */
    public static boolean testMillerRabin(long p, int iters) {
        long a = p - 1;
        long b = 0;
        long c;
        while ((a & 1) == 0) {
            a >>= 1;
            ++b;
        }
        for (int i = 0; i < iters; ++i) {
            c = MathUtil.randomInRange(0, p);
            if (!millerRabinPomoc(a, b, c, p)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Metoda pomocnicza dla testu Millera-Rabina
     */
    private static boolean millerRabinPomoc(long a, long b, long c, long d) {
        long h = 0;
        long e = d - 1;
        long f = b - 1;
        long g = powerMod(c, a, d);
        if (g == 1) {
            return true;
        }
        while (g != e) {
            if (h > f) {
                return false;
            }
            g = powerMod(g, 2, d);
            if (g == 1) {
                return false;
            }
        }
        return true;
    }

    /**
     * oblicza liczby pierwsze w podanym zakresie
     *
     * @param min int - pierwsza liczba
     * @param max int - większa liczba
     * @return ArrayList<Integer> - arraylista liczb pierwszych
     * leżących między min, a max
     */
    public static ArrayList<Integer> getPrimes(int min, int max) {
        ArrayList<Integer> list = new ArrayList<>();
        for (int j = min + 1; j < max; j++) {
            if (isPrime(j)) {
                list.add(j);
            }
        }
        return list;
    }

    public static Point[] factorize1(int n) {
        FrequencyMap<Integer> al = new FrequencyMap<>();
        int i = 2;
        int e = (int) Math.floor(Math.sqrt(n));
        while (i <= e) {
            if (n % i == 0) {
                al.add(i);
                n = n / i;
                e = (int) Math.floor(Math.sqrt(n));
            } else {
                i++;
            }
        }
        if (n > 1) {
            al.add(n);
        }
        return al.getAllPoints();
    }

    public static Tuple2L[] factorize1(long n) {
        FrequencyMap<Long> al = new FrequencyMap<>();
        long i = 2;
        long e = (long) Math.floor(Math.sqrt(n));
        while (i <= e) {
            if (n % i == 0) {
                al.add(i);
                n = n / i;
                e = (long) Math.floor(Math.sqrt(n));
            } else {
                i++;
            }
        }
        if (n > 1) {
            al.add(n);
        }
        return al.getAllAsTuplesL();
    }

    public static int[] factorize2(int n) {
        FrequencyMap<Integer> al = new FrequencyMap<>();
        int i = 2;
        int e = (int) Math.floor(Math.sqrt(n));
        while (i <= e) {
            if (n % i == 0) {
                al.add(i);
                n = n / i;
                e = (int) Math.floor(Math.sqrt(n));
            } else {
                i++;
            }
        }
        if (n > 1) {
            al.add(n);
        }
        return al.getAllAsInts();
    }

    public static long[] factorize2(long n) {
        FrequencyMap<Long> al = new FrequencyMap<>();
        long i = 2;
        long e = (long) Math.floor(Math.sqrt(n));
        while (i <= e) {
            if (n % i == 0) {
                al.add(i);
                n = n / i;
                e = (long) Math.floor(Math.sqrt(n));
            } else {
                i++;
            }
        }
        if (n > 1) {
            al.add(n);
        }
        return al.getAllAsLongs();
    }
    //------------------------------- inne ----------------------------------

    /**
     * Wykonuje mnożenie modularne
     *
     * @param a liczba q
     * @param b the liczba b
     * @param m the modulus
     * @return a*b (mod m))
     * faktycznie funkcja wykonuje mnożenie a*b = c
     * potem oblicza c%m, ale ta metoda wykonuje to szybciej
     */
    public static long multMod(long a, long b, long m) {
        long wynik = 0;
        while (a > 0) {
            if ((a & 1) == 1) {
                wynik += b;
                wynik %= m;
            }
            b <<= 1;
            b %= m;
            a >>= 1;
        }
        return wynik;
    }

    /**
     * Wykonuje potęgowanie modularne
     *
     * @param a liczba a
     * @param b liczba b
     * @param m modulus
     * @return wynik potęgowania modularnego. Właściwie
     * funkcja podnosi a do potęgi b = c
     * oblicza c%modulus, ale wykonuje to szybciej
     */
    public static long powerMod(long a, long b, long m) {
        long wynik = 1;
        while (b > 0) {
            if ((b & 1) == 1) {
                wynik = multMod(wynik, a, m);
            }
            a = multMod(a, a, m);
            b >>= 1;
        }
        return wynik;
    }

    public static boolean[] sitoEratostenesa2(int n) {
        boolean[] tabl = new boolean[n + 1];
        Arrays.fill(tabl, true);
        tabl[0] = false;
        tabl[1] = false;
        int nn = (int) Math.floor(Math.sqrt(n));
        int k = 2;
        while (k <= nn) {
            for (int i = 2 * k; i < n + 1; i = i + k) {//usuwamy 2
                tabl[i] = false;
            }
            for (int j = k + 1; j < n + 1; j++) {
                if (tabl[j]) {
                    k = j;
                    break;
                }
            }
        }
        return tabl;
    }

    public static void close(PrintWriter pw) {
        if (pw != null) {
            pw.close();
        }
    }
}
