// Example 10-7: OfflineLockManager.java

package com.oreilly.patterns.chapter10;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;


public class OfflineLockManager {
 private DataSource dataSource;
 private static final String LOCK_INSERT_STMT =
    "INSERT INTO LOCK_TRACKING "+
    "(OBJECT_TYPE, OBJECT_KEY, USERNAME) VALUES (?, ?, ?)";

 private static final String LOCK_SELECT_STMT =
    "SELECT OBJECT_TYPE, OBJECT_KEY, USERNAME, " + 
    "OBTAINED FROM LOCK_TRACKING WHERE " + 
    "OBJECT_TYPE = ? AND OBJECT_KEY = ?";

 private static final String RELEASE_LOCK_STMT =
    "DELETE FROM LOCK_TRACKING WHERE OBJECT_TYPE = ? "+
    "AND OBJECT_KEY = ? AND USERNAME = ?";

 private static final String RELEASE_USER_LOCKS_STMT =
    "DELETE FROM LOCK_TRACKING WHERE USERNAME = ?";

 // polecenia zwolnienia blokady specyficzne dla bazy Oracle; 
 // zwalnia wszystkie blokady po 15 minutach (1/96 doby)
 private static final String RELEASE_AGED_LOCKS_STMT =
    "DELETE FROM LOCK_TRACKING WHERE OBTAINED < SYSDATE - (1/96)";

 public OfflineLockManager(DataSource ds) {
   dataSource = ds;
 }

 public boolean getLock(String objectType, long key, String username)
         throws LockingException {

     Connection con = null;
     PreparedStatement pstmt = null;
     boolean gotLock = false;

     try {
       con = dataSource.getConnection();
       // uywa penej izolacji
       con.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
       con.setAutoCommit(false);
       pstmt = con.prepareStatement(LOCK_INSERT_STMT);
       pstmt.setString(1, objectType);
       pstmt.setLong(2, key);
       pstmt.setString(3, username);

       try {
          pstmt.executeUpdate();
          gotLock = true;
       } catch (SQLException ex) {
       } // wyjtek SQLException oznacza naruszenie unikalnoci klucza
         // gwnego czyli istniejc ju blokad
       if (!gotLock) { 
          // kto posiada ju blokad tego obiektu!
          String lockingUsername = getLockingUser(con, objectType, key);
          if ((lockingUsername != null) && (lockingUsername.equals(username)))
             gotLock = true; // Sami mamy ju t blokad!
       }

       con.commit(); // koniec transakcji
     } catch (SQLException e) {
       try {
         con.rollback();
       } catch (SQLException ignored) {}
       LockingException le = new LockingException(e.getMessage());
       le.initCause(e); // JDK 1.4; usun dla wczeniejszych wersji JDK 
       throw le;
     } finally {
        if (pstmt != null) 
            try { pstmt.close(); } catch (SQLException ignored) {}
        if (con != null) 
            try { con.close(); } catch (SQLException ignored) {}
     }

     return gotLock;
 }

 /**
 * Zwalnia blokad posiadan przez podanego uytkownika dla podanej
 * pary typ/klucz.
 */
 public boolean releaseLock(String objectType, long key, String username)
         throws LockingException {
   Connection con = null;
   PreparedStatement pstmt = null;
   try {
     con = dataSource.getConnection();
     pstmt = con.prepareStatement(RELEASE_LOCK_STMT);
     pstmt.setString(1, objectType);
     pstmt.setLong(2, key);
     pstmt.setString(3, username);
     int count = pstmt.executeUpdate();
     return (count > 0); // jeli rekord zosta usunity, 
                         // to blokada jest zwolniona
   } catch (SQLException e) {
     LockingException le = new LockingException(e.getMessage());
     le.initCause(e); // JDK 1.4; usun dla wczeniejszych wersji JDK 
     throw le;
   } finally {
     if (pstmt != null) 
         try { pstmt.close(); } catch (SQLException ignored) {}
     if (con != null) 
         try { con.close(); } catch (SQLException ignored) {}
   }
 }

 /**
  * Zwalnia wszystkie blokady nalece do podanego uytkownika. 
  * Zwraca true, gdy blokady zostay usunite.
  */
 public boolean releaseUserLocks(String username) throws LockingException {
   Connection con = null;
   PreparedStatement pstmt = null;
   try {
     con = dataSource.getConnection();
     pstmt = con.prepareStatement(RELEASE_USER_LOCKS_STMT);
     pstmt.setString(1, username);
     int count = pstmt.executeUpdate();
     return (count > 0); // jeli rekordy zostay usunite, 
                         // to blokady s zwolnione.
   } catch (SQLException e) {
     LockingException le = new LockingException(e.getMessage());
     le.initCause(e); // JDK 1.4; usun dla wczeniejszych wersji JDK 
     throw le;
   } finally {
     if (pstmt != null) 
         try { pstmt.close(); } catch (SQLException ignored) {}
     if (con != null) 
         try { con.close(); } catch (SQLException ignored) {}
   }
 }

 /**
   * Zwalnia wszystkie blokady starsze ni 15 minut.
   */
 public boolean releaseAgedLocks() throws LockingException {
   Connection con = null;
   PreparedStatement pstmt = null;

   try {
      con = dataSource.getConnection();
      pstmt = con.prepareStatement(RELEASE_AGED_LOCKS_STMT);
      int count = pstmt.executeUpdate();
      return (count > 0); // jeli rekordy zostay usunite, 
                          // to blokady s zwolnione.
   } catch (SQLException e) {
      LockingException le = new LockingException(e.getMessage());
      le.initCause(e); // JDK 1.4; usun dla wczeniejszych wersji JDK 
      throw le;
   } finally {
      if (pstmt != null) 
          try { pstmt.close(); } catch (SQLException ignored) {}
      if (con != null) 
          try {  con.close(); } catch (SQLException ignored) {}
   }
 }


  /**
    * Zwraca uytkownika bdcego w posiadaniu blokady dla pary typ/klucz, 
    * lub null, gy nie ma blokady.
    */
  private String getLockingUser(Connection con, String objectType, 
                                long key) throws SQLException {
    PreparedStatement pstmt = null;
    try {
      pstmt = con.prepareStatement(LOCK_SELECT_STMT);
      pstmt.setString(1, objectType);
      pstmt.setLong(2, key);
      ResultSet rs = pstmt.executeQuery();
      String lockingUser = null;
      if (rs.next())
        lockingUser = rs.getString("USERNAME");
        rs.close();
        return lockingUser;
    } catch (SQLException e) {
      throw e;
    } finally {
      if (pstmt != null) 
          try { pstmt.close(); } catch (SQLException ignored) {}
    }
 }

}
