package apexample.model;

import java.util.*;
import java.io.IOException;

/**
 * Klasa zarzdzajca subskrypcjami stron RSS
 */
public class RSSSubscriber extends Thread {
    /** Domylny okres aktualizacji.  30 sekund */
    private static final int UPDATE_FREQ = 30 * 1000;
    
    /**
     * Wewntrzna reprezentacja subskrypcji
     */
    class RSSSubscription implements Comparable {
        /** adres url rda RSS */
        private String url;
        
        /** czas nastpnej aktualizacji */
        private long nextUpdate;
        
        /** czstotliwo aktualizacji dla tej subskrypcji */
        private long updateFreq;
       
        
        /** 
         * Porwnanie z inn subskrypcj. Sortowanie subskrybcji
         * w oparciu o czas nastpnej aktualizacji
         */
        public int compareTo(Object obj) {
            RSSSubscription rObj = (RSSSubscription) obj;
            if (rObj.nextUpdate > this.nextUpdate) {
                return -1;
            } else if (rObj.nextUpdate < this.nextUpdate) {
                return 1;
            } else {
                // jeli czas aktualizacji jest taki sam, 
                // to porwnuje adresy URL
                return url.compareToIgnoreCase(rObj.url);
            }
        }
    }
    
    /** zbir subskrybcji posortowany na podstawie czasu kolejnej aktualizacji*/
    private SortedSet subscriptions;
    
    /** istniejce subskrybcje */
    private Map cache;
    
    /** zakoczenie */
    private boolean quit = false;
    
    /** singleton subskrybenta */
    private static RSSSubscriber subscriber;
    
    /* inicjacja singletonu */
    static {
        subscriber = new RSSSubscriber();
        subscriber.start();
    }
    
    /**
     * Zwraca referencj subskrybenta.  Uywana 
     * zamiast konstruktora.
     */
    public static RSSSubscriber getInstance() {
        return subscriber;
    }
    
    /**
     * Konstruktor.  Inicjuje struktury danych.
     */
    RSSSubscriber() {
        subscriptions = new TreeSet();
        cache = Collections.synchronizedMap(new HashMap());
        
        // koniec, gdy jest to jedyny istniejcy wtek
        setDaemon(true);
    }
    
    /**
     * Zwraca obiekt RSSInfo z bufora lub tworzy now
     * subskrypcj.
     */
    public RSSInfo getInfo(String url) throws Exception {
        if (cache.containsKey(url)) {
            return (RSSInfo)cache.get(url);
        }

        // dodaje do bufora
        RSSInfo rInfo = new RSSInfo();
        rInfo.parse(url);
        cache.put(url, rInfo);
        
        // tworzy now subskrypcj
        RSSSubscription newSub = new RSSSubscription();
        newSub.url = url;
        newSub.updateFreq = UPDATE_FREQ;
        putSubscription(newSub);
        
        return rInfo;
    }

    /**
     * Dodaje subskrypcj
     */
    private synchronized void putSubscription(RSSSubscription subs) {
        subs.nextUpdate = System.currentTimeMillis() + subs.updateFreq;
        subscriptions.add(subs);
        notify();
    }
    
    /**
     * Oczekuje a kolejna subskrypcja bdzie gotowa do aktualizacji
     */
    private synchronized RSSSubscription getSubscription() {
        while(true) {
            while(subscriptions.size() == 0) {
                try { wait(); } catch(InterruptedException ie) {}
            }
            
            RSSSubscription nextSub = (RSSSubscription)subscriptions.first();
               
            long curTime = System.currentTimeMillis();
            if(curTime >= nextSub.nextUpdate) {
                subscriptions.remove(nextSub);
                return nextSub;
            }
            
            try {
                wait(nextSub.nextUpdate - curTime);
            } catch(InterruptedException ie) {}
        }
    }
    
    /**
     * Aktualizuje gotow subskrybcj
     */
    public void run() {
        while(!quit) {
            RSSSubscription subs = getSubscription();
            
            try {
                RSSInfo rInfo = new RSSInfo();
                rInfo.parse(subs.url);
                cache.put(subs.url, rInfo);
            } catch(Exception ex) {
                ex.printStackTrace();
            }
            
            putSubscription(subs);
        }
    }
    
    /**
     * Zatrzymuje subskrybenta
     */
    public void quit() {
        this.quit = true;
    }
}

