/*
 Erica Sadun, http://ericasadun.com
 iPhone Developer's Cookbook, 5.0 Edition
 BSD License for anything not specifically marked as developed by a third party.
 Apple's code excluded.
 Use at your own risk
 */

#import <SystemConfiguration/SystemConfiguration.h>

#import <arpa/inet.h>
#import <netdb.h>
#import <net/if.h>
#import <ifaddrs.h>
#import <unistd.h>
#import <dlfcn.h>
#import <notify.h>

#import "UIDevice-Reachability.h"
#import "wwanconnect.h"

@implementation UIDevice (Reachability)
SCNetworkConnectionFlags connectionFlags;
SCNetworkReachabilityRef reachability;

#pragma mark Klasa IP i metody narzędziowe 
// Poniższe narzędzia IP są zainspirowane lub powstały na podstawie kodu Apple. Dziękuję Ci Apple.

+ (NSString *) stringFromAddress: (const struct sockaddr *) address
{
	if (address && address->sa_family == AF_INET) 
    {
		const struct sockaddr_in* sin = (struct sockaddr_in *) address;
		return [NSString stringWithFormat:@"%@:%d", [NSString stringWithUTF8String:inet_ntoa(sin->sin_addr)], ntohs(sin->sin_port)];
	}
	
	return nil;
}

+ (BOOL)addressFromString:(NSString *)IPAddress address:(struct sockaddr_in *)address
{
	if (!IPAddress || ![IPAddress length]) return NO;
	
	memset((char *) address, sizeof(struct sockaddr_in), 0);
	address->sin_family = AF_INET;
	address->sin_len = sizeof(struct sockaddr_in);
	
	int conversionResult = inet_aton([IPAddress UTF8String], &address->sin_addr);
	if (conversionResult == 0) 
    {
		NSAssert1(conversionResult != 1, @"Nie udało się skonwertować ciągu tekstowego adresu IP na sockaddr_in: %@", IPAddress);
		return NO;
	}
	
	return YES;
}

+ (NSString *) addressFromData:(NSData *) addressData
{
    NSString *adr = nil;
	
    if (addressData != nil)
    {
		struct sockaddr_in addrIn = *(struct sockaddr_in *)[addressData bytes];
		adr = [NSString stringWithFormat: @"%s", inet_ntoa(addrIn.sin_addr)];
    }
	
    return adr;
}

+ (NSString *) portFromData:(NSData *) addressData
{
    NSString *port = nil;
	
    if (addressData != nil)
    {
		struct sockaddr_in addrIn = *(struct sockaddr_in *)[addressData bytes];
		port = [NSString stringWithFormat: @"%s", ntohs(addrIn.sin_port)];
    }
	
    return port;
}

+ (NSData *) dataFromAddress: (struct sockaddr_in) address
{
	return [NSData dataWithBytes:&address length:sizeof(struct sockaddr_in)];
}

- (NSString *) hostname
{
	char baseHostName[256]; // Podziękowania dla Gunnara Larischa.
	int success = gethostname(baseHostName, 255);
	if (success != 0) return nil;
	baseHostName[255] = '\0';
	
#if TARGET_IPHONE_SIMULATOR
 	return [NSString stringWithFormat:@"%s", baseHostName];
#else
	return [NSString stringWithFormat:@"%s.local", baseHostName];
#endif
}

- (NSString *) getIPAddressForHost: (NSString *) theHost
{
	struct hostent *host = gethostbyname([theHost UTF8String]);
    if (!host) {herror("resolv"); return NULL; }
	struct in_addr **list = (struct in_addr **)host->h_addr_list;
	NSString *addressString = [NSString stringWithCString:inet_ntoa(*list[0]) encoding:NSUTF8StringEncoding];
	return addressString;
}

- (NSString *) localIPAddress
{
	struct hostent *host = gethostbyname([[self hostname] UTF8String]);
    if (!host) {herror("resolv"); return nil;}
    struct in_addr **list = (struct in_addr **)host->h_addr_list;
	return [NSString stringWithCString:inet_ntoa(*list[0]) encoding:NSUTF8StringEncoding];
}

// Matt Brown dostarczył rozwiązanie sprawdzania adresu IP połączenia WiFi.
// Matt pozwolił na użycie kodu w niniejszej książce na takiej samej licencji jak pozostały kod.
// http://mattbsoftware.blogspot.com/2009/04/how-to-get-ip-address-of-iphone-os-v221.html

// Hotspot iPhone dzięki uprzejmości Johannesa Rudolpha.

- (NSString *) localWiFiIPAddress
{
	BOOL success;
	struct ifaddrs * addrs;
	const struct ifaddrs * cursor;
	
	success = getifaddrs(&addrs) == 0;
	if (success) {
		cursor = addrs;
		while (cursor != NULL) {
            
            // Drugi test pozwala uniknąć adresu loopback.
			if (cursor->ifa_addr->sa_family == AF_INET && (cursor->ifa_flags & IFF_LOOPBACK) == 0) 
			{
				NSString *name = [NSString stringWithUTF8String:cursor->ifa_name];

                /*
                 // Usuń znaki komentarza podczas debugowania.
                 NSLog(@"Nazwa interfejsu: %@, inet: %d, loopback: %d, adres: %@", 
                 name, 
                 cursor->ifa_addr->sa_family == AF_INET, 
                 (cursor->ifa_flags & IFF_LOOPBACK) == 0, 
                 [NSString stringWithUTF8String:inet_ntoa(((struct sockaddr_in *)cursor->ifa_addr)->sin_addr)]);
                 */                

                // Adapter Wi-Fi lub adapter pomostu osobistego hotspotu.
				if ([name isEqualToString:@"en0"] || 
                    [name isEqualToString:@"bridge0"])
					return [NSString stringWithUTF8String:inet_ntoa(((struct sockaddr_in *)cursor->ifa_addr)->sin_addr)];
			}
			cursor = cursor->ifa_next;
		}
		freeifaddrs(addrs);
	}
	return nil;
}


- (NSArray *) localWiFiIPAddresses
{
	BOOL success;
	struct ifaddrs * addrs;
	const struct ifaddrs * cursor;
	
	NSMutableArray *array = [NSMutableArray array];
	
	success = getifaddrs(&addrs) == 0;
	if (success) {
		cursor = addrs;
		while (cursor != NULL) {
            
			// Drugi test pozwala uniknąć adresu loopback.
			if (cursor->ifa_addr->sa_family == AF_INET && (cursor->ifa_flags & IFF_LOOPBACK) == 0) 
			{
				NSString *name = [NSString stringWithUTF8String:cursor->ifa_name];
                
                // Adapter Wi-Fi lub adapter pomostu osobistego hotspotu.
				if ([name hasPrefix:@"en"] || [name hasPrefix:@"bridge"])
					[array addObject:[NSString stringWithUTF8String:inet_ntoa(((struct sockaddr_in *)cursor->ifa_addr)->sin_addr)]];
			}
			cursor = cursor->ifa_next;
		}
		freeifaddrs(addrs);
	}
	
	if (array.count) return array;
	
	return nil;
}

- (NSString *) whatismyipdotcom
{
	NSError *error;
    NSURL *ipURL = [NSURL URLWithString:@"http://automation.whatismyip.com/n09230945.asp"];
    NSString *ip = [NSString stringWithContentsOfURL:ipURL encoding:1 error:&error];
	return ip ? ip : [error localizedDescription];
}

- (BOOL) hostAvailable: (NSString *) theHost
{
	
    NSString *addressString = [self getIPAddressForHost:theHost];
    if (!addressString)
    {
        NSLog(@"Błąd podczas pobierania adresu IP dla danego urządzenia\n");
        return NO;
    }
	
    struct sockaddr_in address;
    BOOL gotAddress = [UIDevice addressFromString:addressString address:&address];
	
    if (!gotAddress)
    {
		NSLog(@"Błąd podczas określania adresu na podstawie %@", addressString);
        return NO;
    }
	
	SCNetworkReachabilityRef defaultRouteReachability = SCNetworkReachabilityCreateWithAddress(NULL, (struct sockaddr *)&address);
    SCNetworkReachabilityFlags flags;
	
	BOOL didRetrieveFlags = SCNetworkReachabilityGetFlags(defaultRouteReachability, &flags);
    CFRelease(defaultRouteReachability);
	
    if (!didRetrieveFlags)
    {
        NSLog(@"Błąd. Nie można pobrać opcji.");
        return NO;
    }
	
    BOOL isReachable = flags & kSCNetworkFlagsReachable;
    return isReachable ? YES : NO;;
}

#pragma mark Sprawdzenie połączeń
- (void) pingReachabilityInternal
{
	if (!reachability)
	{
		BOOL ignoresAdHocWiFi = NO;
		struct sockaddr_in ipAddress;
		bzero(&ipAddress, sizeof(ipAddress));
		ipAddress.sin_len = sizeof(ipAddress);
		ipAddress.sin_family = AF_INET;
		ipAddress.sin_addr.s_addr = htonl(ignoresAdHocWiFi ? INADDR_ANY : IN_LINKLOCALNETNUM);

		/* Może również utworzyć adres zerowy
		 struct sockaddr_in zeroAddress;
		 bzero(&zeroAddress, sizeof(zeroAddress));
		 zeroAddress.sin_len = sizeof(zeroAddress);
		 zeroAddress.sin_family = AF_INET; */
		
		reachability = SCNetworkReachabilityCreateWithAddress(kCFAllocatorDefault, (struct sockaddr *)&ipAddress);
		CFRetain(reachability);
	}
	
	// Opcje dotyczące połączenia.
	BOOL didRetrieveFlags = SCNetworkReachabilityGetFlags(reachability, &connectionFlags);
	if (!didRetrieveFlags) printf("Błąd. Nie można pobrać opcji\n");
}

- (BOOL) networkAvailable
{
	[self pingReachabilityInternal];
	BOOL isReachable = ((connectionFlags & kSCNetworkFlagsReachable) != 0);
    BOOL needsConnection = ((connectionFlags & kSCNetworkFlagsConnectionRequired) != 0);
    return (isReachable && !needsConnection) ? YES : NO;
}

// Podziękowania dla Johannesa Rudolpha.
- (BOOL) activePersonalHotspot
{
    // Adres osobistego hotspotu został na stałe określony jako 172.20.10.x.
    NSString* localWifiAddress = [self localWiFiIPAddress];
    return (localWifiAddress != nil && [localWifiAddress hasPrefix:@"172.20.10"]);
}


- (BOOL) activeWWAN
{
	if (![self networkAvailable]) return NO;
	return ((connectionFlags & kSCNetworkReachabilityFlagsIsWWAN) != 0);
}

- (BOOL) activeWLAN
{
	return ([[UIDevice currentDevice] localWiFiIPAddress] != nil);
}

#pragma mark Sprawdzenie połączenia WiFi i wyświetlenie komunikatu
- (void) privateShowAlert: (id) formatstring,...
{
	va_list arglist;
	if (!formatstring) return;
	va_start(arglist, formatstring);
	id outstring = [[NSString alloc] initWithFormat:formatstring arguments:arglist];
	va_end(arglist);
	
    UIAlertView *av = [[UIAlertView alloc] initWithTitle:outstring message:nil delegate:nil cancelButtonTitle:@"OK"otherButtonTitles:nil];
	[av show];
}

- (BOOL) performWiFiCheck
{
	if (![self networkAvailable] || ![self activeWLAN])
	{
		[self performSelector:@selector(privateShowAlert:) withObject:@"Aplikacja wymaga połączenia WiFi. Proszę włączyć WiFi w aplikacji Ustawienia, a następnie ponownie uruchomić tę aplikację." afterDelay:0.5f];
		return NO;
	}
	return YES;
}

#pragma mark Wymuszenie połączenia WWAN. Dzięki uprzejmości Apple. Dziękuję Ci Apple
MyStreamInfoPtr	myInfoPtr;
static void myClientCallback(void *refCon)
{
	int  *val = (int*)refCon;
	printf("wprowadzono myClientCallback - wartość z refCon to %d\n", *val);
}

- (BOOL) forceWWAN
{
	int value = 0;
	myInfoPtr = (MyStreamInfoPtr) StartWWAN(myClientCallback, &value);
	NSLog(@"%@", myInfoPtr ? @"Uruchomiono WWAN" : @"Nie udało się uruchomić WWAN");
	return (!(myInfoPtr == NULL));
}

- (void) shutdownWWAN
{
	if (myInfoPtr) StopWWAN((MyInfoRef) myInfoPtr);
}

#pragma mark Monitorowanie dostępności
static void ReachabilityCallback(SCNetworkReachabilityRef target, SCNetworkConnectionFlags flags, void* info)
{
    @autoreleasepool {
        id watcher = (__bridge id) info;
        SEL changed = @selector(reachabilityChanged);
        if ([watcher respondsToSelector:changed])
            [watcher performSelector:changed];
    }
}

- (BOOL) scheduleReachabilityWatcher: (id) watcher
{
    SEL changed = @selector(reachabilityChanged);
	if (![watcher respondsToSelector:changed])
	{
		NSLog(@"Błąd: obserwator musi implementować metodę reachabilityChanged.");
		return NO;
	}
	
	[self pingReachabilityInternal];

	SCNetworkReachabilityContext context = {0, (__bridge void *)watcher, NULL, NULL, NULL};
	if(SCNetworkReachabilitySetCallback(reachability, ReachabilityCallback, &context)) 
	{
		if(!SCNetworkReachabilityScheduleWithRunLoop(reachability, CFRunLoopGetCurrent(), kCFRunLoopCommonModes)) 
		{
			NSLog(@"Błąd: nie można ustawić obserwatora.");
			SCNetworkReachabilitySetCallback(reachability, NULL, NULL);
			return NO;
		}
	} 
	else 
	{
		NSLog(@"Błąd: nie można zdefiniować wywołania zwrotnego obserwatora.");
		return NO;
	}
	
	return YES;
}

- (void) unscheduleReachabilityWatcher
{
	SCNetworkReachabilitySetCallback(reachability, NULL, NULL);
	if (SCNetworkReachabilityUnscheduleFromRunLoop(reachability, CFRunLoopGetCurrent(), kCFRunLoopCommonModes))
		NSLog(@"Sukces. Udało się usunąć obserwatora.");
	else
		NSLog(@"Błąd: nie udało się usunąć obserwatora.");
	
	CFRelease(reachability);
	reachability = nil;
}
@end