/*
 Erica Sadun, http://ericasadun.com
 iPhone Developer's Cookbook, 5.x Edition
 BSD License, Use at your own risk
 */

#import <UIKit/UIKit.h>

#define COOKBOOK_PURPLE_COLOR	[UIColor colorWithRed:0.20392f green:0.19607f blue:0.61176f alpha:1.0f]
#define BARBUTTON(TITLE, SELECTOR) 	[[UIBarButtonItem alloc] initWithTitle:TITLE style:UIBarButtonItemStylePlain target:self action:SELECTOR]

#define POINTSTRING(_CGPOINT_) (NSStringFromCGPoint(_CGPOINT_))
#define VALUE(_INDEX_) [NSValue valueWithCGPoint:points[_INDEX_]]
#define POINT(_INDEX_) [(NSValue *)[points objectAtIndex:_INDEX_] CGPointValue]

#pragma mark Narzędzia krzywych Beziera
// Pobranie punktów z krzywej Béziera.
void getPointsFromBezier(void *info, const CGPathElement *element) 
{
    NSMutableArray *bezierPoints = (__bridge NSMutableArray *)info;    
    
    // Pobranie typu elementu ścieżki i jego punktów.
    CGPathElementType type = element->type;
    CGPoint *points = element->points;
    
    // Dodanie punktów, o ile są dostępne  (dla poszczególnych typów).
    if (type != kCGPathElementCloseSubpath)
    {
        [bezierPoints addObject:VALUE(0)];
        if ((type != kCGPathElementAddLineToPoint) &&
            (type != kCGPathElementMoveToPoint))
            [bezierPoints addObject:VALUE(1)];
    }    
    if (type == kCGPathElementAddCurveToPoint)
        [bezierPoints addObject:VALUE(2)];
}

NSArray *pointsFromBezierPath(UIBezierPath *bpath)
{
    NSMutableArray *points = [NSMutableArray array];
    CGPathApply(bpath.CGPath, (__bridge void *)points, getPointsFromBezier);
    return points;
}

#pragma mark Narzędzia geometrii
// Zwrot punktu środkowego podanego prostokąta.
CGPoint getRectCenter(CGRect rect)
{
	return CGPointMake(CGRectGetMidX(rect), CGRectGetMidY(rect));
}

// Utworzenie prostokąta wokół danego punktu środkowego.
CGRect rectAroundCenter(CGPoint center, float dx, float dy)
{
	return CGRectMake(center.x - dx, center.y - dy, dx * 2, dy * 2);
}

// Zwrot wartości dwóch znormalizowanych wektorów.
float dotproduct (CGPoint v1, CGPoint v2)
{
	float dot = (v1.x * v2.x) + (v1.y * v2.y);
	float a = ABS(sqrt(v1.x * v1.x + v1.y * v1.y));
	float b = ABS(sqrt(v2.x * v2.x + v2.y * v2.y));
	dot /= (a * b);
	
	return dot;
}

// Zwrot odległości pomiędzy dwoma punktami.
float distance (CGPoint p1, CGPoint p2)
{
	float dx = p2.x - p1.x;
	float dy = p2.y - p1.y;
	
	return sqrt(dx*dx + dy*dy);
}

// Zwrot punktu z uwzględnieniem podanego punktu początkowego.
CGPoint pointWithOrigin(CGPoint pt, CGPoint origin)
{
	return CGPointMake(pt.x - origin.x, pt.y - origin.y);
}

#pragma mark Wykrycie okręgu

// Obliczenie i zwrot prostokąta ograniczającego.
CGRect boundingRect(NSArray *points)
{
	CGRect rect = CGRectZero;
	CGRect ptRect;
	
	for (int i = 0; i < points.count; i++)
	{
        CGPoint pt = POINT(i);
		ptRect = CGRectMake(pt.x, pt.y, 0.0f, 0.0f);
		rect = (CGRectEqualToRect(rect, CGRectZero)) ? ptRect : CGRectUnion(rect, ptRect);
	}
	
	return rect;
}

#define DX(p1, p2)	(p2.x - p1.x)
#define DY(p1, p2)	(p2.y - p1.y)
#define SIGN(NUM) (NUM < 0 ? (-1) : 1)
#define DEBUG NO

CGRect testForCircle(NSArray *points, NSDate *firstTouchDate)
{
	if (points.count < 2) 
	{
		if (DEBUG) NSLog(@"Zbyt mało punktów (2) jak na okrąg");
		return CGRectZero;
	}
	
	// Test 1: tolerancja czasu trwania.
	float duration = [[NSDate date] timeIntervalSinceDate:firstTouchDate];
	if (DEBUG) NSLog(@"Czas trwania przejścia: %0.2f", duration);
        
        float maxDuration = 2.0f;
        if (duration > maxDuration) // W symulatorze można użyć większej wartości.
        {
            if (DEBUG) NSLog(@"Przekroczony czas trwania: %0.2f sekund vs %0.1f sekund", duration, maxDuration);
            return CGRectZero;
        }
	
	// Test 2: liczba zmian kierunków, która powinna być ograniczona do około czterech.
	int inflections = 0;
	for (int i = 2; i < (points.count - 1); i++)
	{
		float dx = DX(POINT(i), POINT(i-1));
		float dy = DY(POINT(i), POINT(i-1));
		float px = DX(POINT(i-1), POINT(i-2));
		float py = DY(POINT(i-1), POINT(i-2));
		
		if ((SIGN(dx) != SIGN(px)) || (SIGN(dy) != SIGN(py)))
			inflections++;
	}
	
	if (inflections > 5)
	{
		if (DEBUG) NSLog(@"Przekroczono liczbę zmian kierunków (%d vs 4). Porażka.", inflections);
		return CGRectZero;
	}
	
	// Test 3: Punkty początkowy i końcowy muszą znajdować
    // się w pewnej odległości, nie większej niż zdefiniowana.
	float tolerance = [[[UIApplication sharedApplication] keyWindow] bounds].size.width / 3.0f;	
	if (distance(POINT(0), POINT(points.count - 1)) > tolerance)
	{
		if (DEBUG) NSLog(@"Punkty początkowy i końcowy są zbyt oddalone od siebie. Porażka.");
		return CGRectZero;
	}
	
	// Test 4: Obliczenie pokonanej odległości wyrażonej w stopniach. 
	CGRect circle = boundingRect(points);
	CGPoint center = getRectCenter(circle);
	float distance = ABS(acos(dotproduct(pointWithOrigin(POINT(0), center), pointWithOrigin(POINT(1), center))));
	for (int i = 1; i < (points.count - 1); i++)
		distance += ABS(acos(dotproduct(pointWithOrigin(POINT(i), center), pointWithOrigin(POINT(i+1), center))));
        
        float transitTolerance = distance - 2 * M_PI;
        
        if (transitTolerance < 0.0f) // Można używać wartości mniejszej niż  2 * PI.
        {
            if (transitTolerance < - (M_PI / 4.0f)) // 45 stopni lub więcej.
            {
                if (DEBUG) NSLog(@"Przejście jest zbyt krótkie, poniżej 315 stopni");
                return CGRectZero;
            }
        }
	
	if (transitTolerance > M_PI) // Dodatkowe 180 stopni.
	{
		if (DEBUG) NSLog(@"Przejście jest zbyt długie, ponad 540 stopni");
		return CGRectZero;
	}
	
	return circle;
}


@interface TouchTrackerView : UIView
{
    UIBezierPath *path;
    NSDate *firstTouchDate;
}
@end

@implementation TouchTrackerView
- (void) touchesBegan:(NSSet *) touches withEvent:(UIEvent *) event
{
    path = [UIBezierPath bezierPath];    
    path.lineWidth = 4.0f;
    
    UITouch *touch = [touches anyObject];
    [path moveToPoint:[touch locationInView:self]];
    
    firstTouchDate = [NSDate date];
}

- (void) touchesMoved:(NSSet *) touches withEvent:(UIEvent *) event
{
    UITouch *touch = [touches anyObject];
    [path addLineToPoint:[touch locationInView:self]];
    [self setNeedsDisplay];
}

- (void) touchesEnded:(NSSet *)touches withEvent:(UIEvent *)event
{
    UITouch *touch = [touches anyObject];
    [path addLineToPoint:[touch locationInView:self]];
    [self setNeedsDisplay];
}

- (void) touchesCancelled:(NSSet *)touches withEvent:(UIEvent *)event
{
    [self touchesEnded:touches withEvent:event];
}

- (void) drawRect:(CGRect)rect
{
    [COOKBOOK_PURPLE_COLOR set];
    [path stroke];
    
    CGRect circle = testForCircle(pointsFromBezierPath(path), firstTouchDate);
    if (!CGRectEqualToRect(CGRectZero, circle))
    {
        [[UIColor redColor] set];
        UIBezierPath *circlePath = [UIBezierPath bezierPathWithOvalInRect:circle];
        circlePath.lineWidth = 6.0f;
        [circlePath stroke];
        
        CGRect  centerBit = rectAroundCenter(getRectCenter(circle), 4.0f, 4.0f);
        UIBezierPath *centerPath = [UIBezierPath bezierPathWithOvalInRect:centerBit];
        [centerPath fill];
    }
}

- (id) initWithFrame:(CGRect)frame
{
    if (self = [super initWithFrame:frame])
        self.multipleTouchEnabled = NO;
    
    return self;
}
@end

@interface TestBedViewController : UIViewController
@end

@implementation TestBedViewController

- (void) action: (id) sender
{
}

- (void) loadView
{
    [super loadView];
    self.view = [[TouchTrackerView alloc] initWithFrame:self.view.frame];
    self.view.backgroundColor = [UIColor whiteColor];
}

- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	return YES;
}
@end

#pragma mark -

#pragma mark Konfiguracja aplikacji
@interface TestBedAppDelegate : NSObject <UIApplicationDelegate>
{
	UIWindow *window;
}
@end
@implementation TestBedAppDelegate
- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(NSDictionary *)launchOptions 
{	
    [application setStatusBarHidden:YES];
	window = [[UIWindow alloc] initWithFrame:[[UIScreen mainScreen] bounds]];
	TestBedViewController *tbvc = [[TestBedViewController alloc] init];
    window.rootViewController = tbvc;
	[window makeKeyAndVisible];
    return YES;
}
@end
int main(int argc, char *argv[]) {
    @autoreleasepool {
        int retVal = UIApplicationMain(argc, argv, nil, @"TestBedAppDelegate");
        return retVal;
    }
}