// dołącz obsługę WiFi w zależności od zastosowanego układu ESP
#if defined(ESP8266)
  #include <ESP8266WiFi.h>       // funkcje sieciowe dla układu ESP8266
  #include <ESP8266WebServer.h>  // serwer WWW z obsługą żądań HTTP GET i POST
#elif defined(ESP32)
  #include <WiFi.h>              // funkcje sieciowe dla układu ESP32
  #include <WebServer.h>         // serwer WWW z obsługą żądań HTTP GET i POST
#endif

#include <EasyDDNS.h>

/*
  Lista obsługiwanych operatorów DDNS:
  - "duckdns"
  - "noip"
  - "dyndns"
  - "dynu"
  - "enom"
  - "all-inkl"
  - "selfhost.de"
  - "dyndns.it"
  - "strato"
  - "freemyip"
  - "afraid.org"
*/

// konfiguracja DDNS
#define DDNS_domain "twojanazwa.ddns.net" // adres domeny u operatora DDNS
#define DDNS_service "noip"               // operator DDNS
#define DDNS_username "nazwa użytkownika" // nazwa użytkownika
#define DDNS_passwd "hasło"               // i hasło u operatora DDNS
#define DDNS_token "token" // token do zarządzania domeną np. u operatora duckdns.org
#define DDNS_localIP true  // czy monitorować zmianę adresu IP w sieci lokalnej?

#define serialSpeed 115200 // prędkość transmisji UART

char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

#include "DHT.h" // dołącz obsługę modułu DHT

/*
D2 - NodeMCU DevKit 0.9(V1)/1.0(V2)/V3 ESP-12/E
D8 - WeMos D1 R1/R2 (ESP8266)
5  - ESP-Mx DevKit (ESP8285)
8  - NodeMCU ESP-C3-12F
14 - WeMos D1 R32 (ESP-WROOM-32)
15 - NodeMCU-32-S2 ESP-12K, ESP-DevKit (ESP-WROOM-32)
*/

#define DHTPIN D2     // numer GPIO dla linii sygnałowej DAT
#define DHTTYPE DHT22 // typ czujnika DHT
/*
#define DHTTYPE DHT11
#define DHTTYPE DHT21 // DHT 21 (AM2301)
#define DHTTYPE DHT22 // DHT 22 (AM2302), AM2321
*/

DHT dht(DHTPIN, DHTTYPE); // obiekt obsługujący czujnik DHT

int board_id = 1; // identyfikator urządzenia

// utwórz obiekt serwera WWW na porcie 80
#if defined(ESP8266)
  ESP8266WebServer Server(80);
#elif defined(ESP32)
  WebServer Server(80);
#endif

// kod HTML strony startowej zapisany w pamięci programu (nie w pamięci RAM)
const char page_index[] PROGMEM = R"=====(
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<title>Mój Smart Home</title>
<style>
body { padding: 2rem; }
h1 { font-size: 2rem; }
h2 { font-size: 1rem; }
.item { font-size: 2.5rem; }
.item .unit { font-size: 1.0rem; margin-left: 5px; }
</style>
</head>
<body>
<h1>Mój Smart Home</h1>
<h2>Pomiar temperatury i wilgotności</h2>
<p class="item">Temperatura <span id="temp">0.0</span><sup class="unit">°C</sup></p>
<p class="item">Wilgotność <span id="hmd">0.0</span><sup class="unit">%</sup></p>
<hr>
<p><button onclick="getData()">Odczyt danych</button></p>
</body>
<script>
function getData(){
 var xhttp = new XMLHttpRequest();
 xhttp.onreadystatechange = function(){
  if (this.readyState == 4 && this.status == 200){
    data = this.responseText;
    data = data.split(',');
    document.getElementById('temp').innerHTML = data[0];
    document.getElementById('hmd').innerHTML = data[1];
  }
 };
 xhttp.open('GET', '/data', true);
 xhttp.send();
};
getData();
</script>
</html>
)=====";

void setup()
{
  Serial.begin(serialSpeed);

  Serial.print(F("Inicjalizacja czujnika DHT ... "));
  dht.begin();
  Serial.println(F("gotowe!"));

  // połącz z punktem dostępowym
  Serial.print(F("Łączę się z siecią WiFi "));
  WiFi.begin(wifi_ssid, wifi_passwd);

  while (WiFi.status() != WL_CONNECTED) { // czekaj na połączenie
    delay(500);
    Serial.print(F("."));
  }

  Serial.println(F(" gotowe!"));

  // uruchom serwer WWW
  Server.begin();
  Serial.println(F("Serwer WWW uruchomiony."));
  Serial.println(F("Otwórz w przeglądarce jeden z poniższych adresów URL:"));
  Serial.print(F("=> http://"));
  Serial.println(WiFi.localIP());
  Serial.print(F("=> http://"));
  Serial.println(DDNS_domain);

  // obsługa adresów URL
  Server.on("/", handleRoot);        // URL strony głównej
  Server.on("/data", handleData);    // URL do odczytu temperatury
  Server.onNotFound(handleNotFound); // obsługa błędu HTTP 404

  // aktywuj obsługę operatora DDNS
  EasyDDNS.service(DDNS_service);
  // dostęp do operatora DDNS w celu uaktualnienia adresu IP po jego zmianie
  // EasyDDNS.client(DDNS_domain, DDNS_token); // w przypadku operatora duckdns.org
  EasyDDNS.client(DDNS_domain, DDNS_username, DDNS_passwd);

  // powiadom o wykryciu nowego adresu IP
  EasyDDNS.onUpdate([&](const char* oldIP, const char* newIP)
  {
    Serial.print(F("EasyDDNS - aktualizacja adresu IP: "));
    Serial.println(newIP);
  });  
}

void loop() 
{
  Server.handleClient(); // czekaj na połączenia klientów

  // sprawdzaj adres IP co 10 sekund, po jego zmianie uaktualnij dane u operatora DDNS
  // DDNS_localIP = true oznacza, że monitorowane będą adresy IP w sieci lokalnej, 
  // np.: zamiast http://192.168.1.116 użyjesz np. przez http://myesp.ddns.net
  EasyDDNS.update(10000, DDNS_localIP);
}

String getTemperatureHumidit(bool debug)
{
  float temp = dht.readTemperature();
  float hmd = dht.readHumidity();

  if (debug)
  {
    Serial.print(F("Odczyt danych z czujnika DHT => T: "));
    Serial.print(temp);
    Serial.print(F("°C, H: "));
    Serial.print(hmd);
    Serial.println(F("%"));
  }

  String data = String(temp) + String(",") + String(hmd);
  return data; // zwróć wartość temperatury i wilgotności
}

// obsługa wywołania URL strony głównej
void handleRoot()
{
  Serial.println(F("=> Odebrano żądanie GET /"));
  Server.send(200, "text/html", page_index);
}

// obsługa wywołania URL do odczytu temperatury
void handleData()
{
  Serial.println(F("=> Odebrano żądanie GET /data"));
  String data = getTemperatureHumidit(true);
  Server.send(200, "text/plain", data);
}

// obsługa błędu HTTP 404
void handleNotFound() 
{
  String message = "404 Not Found\n\n";
  message += "The request ";
  message += Server.uri();
  message += " was not found on this server.\n";
  message += "\nMethod: ";
  message += (Server.method() == HTTP_GET) ? "GET" : "POST";
  message += "\nArguments: ";
  message += Server.args();
  message += "\n";
  for (int i = 0; i < Server.args(); i++)
  {
    message += " " + Server.argName(i) + ": " + Server.arg(i) + "\n";
  }
  Server.send(404, "text/plain", message);
}
