// dołącz obsługę WiFi w zależności od zastosowanego układu ESP
#if defined(ESP8266)
  #include <ESP8266WiFi.h>       // funkcje sieciowe dla układu ESP8266
  #include <ESP8266WebServer.h>  // serwer WWW z obsługą żądań HTTP GET i POST
  #include <ESP8266mDNS.h>       // włącz obsługę multicast DNS 
  #include <ESP8266HTTPUpdateServer.h> // i aktualizacji przez OTA
#elif defined(ESP32)
  #include <WiFi.h>              // funkcje sieciowe dla układu ESP32
  #include <WebServer.h>         // serwer WWW z obsługą żądań HTTP GET i POST
  #include <ESPmDNS.h>           // włącz obsługę multicast DNS
  #include <Update.h>            // i aktualizacji przez OTA
#endif

// utwórz obiekt serwera WWW na porcie 80
#if defined(ESP8266)
  ESP8266WebServer Server(80);
  ESP8266HTTPUpdateServer httpUpdater;
#elif defined(ESP32)
  WebServer Server(80);
#endif

// dostęp do sieci WiFi
char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

// autoryzacja dostępu do strony WWW 
// umożliwiającej aktualizację oprogramowania
bool require_auth = true;   // czy włączyć obsługę HTTP Authentication?
bool http_auth = false;     // czy nazwę użytkownika i hasło wpisano poprawnie?
String http_user = "esp";   // nazwa użytkownika
String http_passwd = "esp"; // hasło 

#define serialSpeed 115200 // prędkość transmisji UART
#define mDNS_host "esp-01" // nazwa urządzenia w sieci lokalnej - http://esp-01.local

// kod HTML strony głównej
static const char pageIndex[] PROGMEM = 
R"(<!doctype html>
<html lang="pl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.2/dist/css/bootstrap.min.css">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.9.1/font/bootstrap-icons.css">
<title>Mój Smart Home</title>
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-dark">
  <div class="container-fluid">
    <a class="navbar-brand" href="#"><i class="bi bi-cpu"></i></a>
    <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarSupportedContent" aria-controls="navbarSupportedContent" aria-expanded="false" aria-label="Toggle navigation">
      <span class="navbar-toggler-icon"></span>
    </button>
    <div class="collapse navbar-collapse" id="navbarSupportedContent">
      <ul class="navbar-nav me-auto mb-2 mb-lg-0">
        <li class="nav-item">
          <a class="nav-link active" aria-current="page" href="/">Odczyty</a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="/update">Aktualizuj</a>
        </li>
      </ul>
   <div class="d-flex text-white">Mój Smart Home</div>
    </div>
  </div>
</nav>
<!-- prezentacja danych: początek -->
<div class="container">
  <div class="px-4 py-5 my-5 text-center">
    <div class="col-lg-6 mx-auto">
      <p class="lead mb-4">Na tej stronie będą prezentowane odczyty z czujnika temperatury i wilgotności DHT.</p>
      <p class="lead mb-4">Możesz zaktualizować oprogramowanie.<br>Użytkownik i hasło: esp</p>
      <div class="d-grid gap-2 d-sm-flex justify-content-sm-center">
        <a class="btn btn-outline-secondary btn-lg px-4" href="/update" role="button">Aktualizuj</a>
      </div>
    </div>
  </div>
</div>
<!-- prezentacja danych: koniec -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.2/dist/js/bootstrap.min.js"></script>
</body>
</html>)";

// kod HTML strony do aktualizacji oprogramowania
static const char pageUpdate[] PROGMEM = 
R"(<!doctype html>
<html lang="pl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.2/dist/css/bootstrap.min.css" rel="stylesheet">
<title>Mój Smart Home</title>
</head>
<body>
<div class="container">
<div class="text-center" style="margin: 50px 0 50px 0;">        
  <h1>Aktualizacja ESP8266/ESP32</h1>
</div>
<form method="POST" enctype="multipart/form-data" id="upload_form">
  <div class="input-group mb-3">      
    <input type="file" accept=".bin,.bin.gz" name="firmware" class="form-control" required="required">
    <button type="submit" class="btn btn-primary upload">Wgraj</button>
  </div>  
</form> 
<div class="progress">
  <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" id="progress" style="width: 0%" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100">Postęp: 0%</div>
</div>  
<div class="alert alert-success" role="alert" id="monit" style="margin-top: 20px; display: none;"></div>
</div>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.1/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.2/dist/js/bootstrap.min.js"></script>
<script>
$(function() {
$('form').submit(function(e){
  $("button.upload").attr("disabled","disabled");
  $("#monit").hide();
  e.preventDefault();
  var form = $('#upload_form')[0];  
  var data = new FormData(form);
  $.ajax({url: '/update', type: 'POST', data: data, contentType: false, processData: false, xhr: function(){
    var xhr = new window.XMLHttpRequest();
    xhr.upload.addEventListener('progress', function(event) {
      if (event.lengthComputable) {
        var perc = Math.round((event.loaded / event.total) * 100);
        $('#progress').html('Postęp: ' + perc + '%');
        $("#progress").css("width", perc + '%').attr("aria-valuenow", perc);
      }
    }, false);
    xhr.upload.addEventListener('load', function(event){
    if (typeof event.target.responseText !== 'undefined' && event.target.responseText.indexOf('error') >= 0) {
        $("#monit").removeClass("alert-success").addClass("alert-danger").html("Coś poszło nie tak :(").show();
        $("button.upload").removeAttr("disabled");
      } else {
        $("#monit").removeClass("alert-danger").addClass("alert-success").html("Oprogramowanie zostało wgrane na płytkę :)<br>Za 15 sekund strona zostanie automatycznie odświeżona.").show();
        window.setTimeout(function(){ document.location="/"; }, 15000);
      }
    }, false);  
    return xhr;
    }, success: function(data, textStatus, jqXHR) {
      console.log(data)
    }, error: function (jqXHR, textStatus, errorThrown) {
      console.log(jqXHR.statusCode + ": " + jqXHR.status)
    }
  }); // $.ajax
}); // $('form')
}); // $(function())
</script>
</body>
</html>)";

void setup()
{
  Serial.begin(serialSpeed);

  // połącz z punktem dostępowym
  Serial.print(F("Łączę się z siecią WiFi "));
  WiFi.begin(wifi_ssid, wifi_passwd);

  while (WiFi.status() != WL_CONNECTED) { // czekaj na połączenie
    delay(500);
    Serial.print(F("."));
  }

  Serial.println(F(" gotowe!"));

  // aktywuj usługę multicast DNS
  if (MDNS.begin(mDNS_host)) { 
    Serial.println(F("Usługa mDNS aktywna."));
  }

  // obsługa wywołania URL strony głównej
  Server.on("/", HTTP_GET, []() { // URL strony głównej
    Serial.println(F("=> Odebrano żądanie GET /"));
    Server.sendHeader("Connection", "close");
    Server.send(200, "text/html", pageIndex);
  });

  // obsługa wywołania URL strony do aktualizacji oprogramowania
  Server.on("/update", HTTP_GET, []() {
    Serial.println(F("=> Odebrano żądanie GET /update"));
    // sprawdź czy wymagana jest autoryzacja użytkownika
    if (require_auth) {
      if(
        http_user != emptyString && http_passwd != emptyString && 
        !Server.authenticate(http_user.c_str(), http_passwd.c_str())
      ) {
        Serial.println(F("=> Odebrano żądanie GET /update => wymagana autoryzacja"));
        return Server.requestAuthentication();
      }
    }
    Server.sendHeader("Connection", "close");
    // prześlij kod HTML do użytkownika    
    Server.send(200, "text/html", pageUpdate);
  });  

  // obsługa danych wysłanych z formularza w trakcie aktualizacji oprogramowania
  Server.on("/update", HTTP_POST, [&]() {
    if(require_auth && !http_auth) { // czy użytkownik ma autoryzację?
      Serial.println(F("=> Odebrano żądanie POST /update => wymagana autoryzacja"));
      return Server.requestAuthentication();
    }
    Server.sendHeader("Connection", "close");
    Server.send(200, "text/plain", (Update.hasError()) ? "FAIL" : "OK");
    ESP.restart();
  }, [&]() {
    // obsługa przesłanych danych - skompilowanego pliku binarnego 
    HTTPUpload& upload = Server.upload();
    
    if (upload.status == UPLOAD_FILE_START) { // przesyłanie danych do pamięci
      Serial.printf("Odbieram plik: %s\n", upload.filename.c_str());

      #if defined(ESP8266)
        WiFiUDP::stopAll();
      #endif

      // czy użytkownik ma autoryzację?
      if (require_auth) {
        http_auth = (
          http_user == emptyString || http_passwd == emptyString || 
          Server.authenticate(http_user.c_str(), http_passwd.c_str())
        );
        if(!http_auth){
          Serial.println(F("Aktualizacja niemożliwa, brak autoryzacji."));
          return;        
        }
      }
      // oblicz maksymalną wielkość programu, który można wgrać
      uint32_t maxSketchSpace = (ESP.getFreeSketchSpace() - 0x1000) & 0xFFFFF000;
      if (!Update.begin(maxSketchSpace)) { // udostępnij całą pamięć
        Update.printError(Serial);
      }
    } else if (upload.status == UPLOAD_FILE_WRITE) {
      // wgrywanie oprogramowania do pamięci układu - flashowanie
      if (Update.write(upload.buf, upload.currentSize) != upload.currentSize) {
        Update.printError(Serial);
      }
    } else if (upload.status == UPLOAD_FILE_END) {
      if (Update.end(true)) { // czy wszystko wgrane?
        Server.send(200, "text/html", F("Gotowe!"));
        Serial.printf("Wielkość: %u bajtów\nRestart modułu ...\n", upload.totalSize);
      } else {
        Update.printError(Serial);
      }
    }
  });

  #if defined(ESP8266)
    httpUpdater.setup(&Server);
  #endif
  Server.begin();

  Serial.println(F("Serwer WWW uruchomiony."));
  Serial.println(F("Otwórz w przeglądarce jeden z poniższych adresów URL:"));
  Serial.print(F("=> http://"));
  Serial.println(WiFi.localIP());
  Serial.print(F("=> http://"));
  Serial.print(mDNS_host);
  Serial.println(F(".local"));  
}

void loop() 
{
  Server.handleClient(); // czekaj na połączenia klientów
  #if defined(ESP8266)
    MDNS.update();       // aktualizuj nazwę urządzenia w sieci lokalnej
  #endif
}
