// dołącz obsługę WiFi w zależności od zastosowanego układu ESP
#if defined(ESP8266)
  #include <ESP8266WiFi.h>
#elif defined(ESP32)
  #include <WiFi.h>
#endif

#define serialSpeed 115200 // prędkość transmisji UART

char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

#include "DHT.h" // dołącz obsługę modułu DHT

/*
D2 - NodeMCU DevKit 0.9(V1)/1.0(V2)/V3 ESP-12/E
D8 - WeMos D1 R1/R2 (ESP8266)
5  - ESP-Mx DevKit (ESP8285)
8  - NodeMCU ESP-C3-12F
14 - WeMos D1 R32 (ESP-WROOM-32)
15 - NodeMCU-32-S2 ESP-12K, ESP-DevKit (ESP-WROOM-32)
*/

#define DHTPIN D2     // numer GPIO dla linii sygnałowej DAT
#define DHTTYPE DHT22 // typ czujnika DHT
/*
#define DHTTYPE DHT11
#define DHTTYPE DHT21 // DHT 21 (AM2301)
#define DHTTYPE DHT22 // DHT 22 (AM2302), AM2321
*/

DHT dht(DHTPIN, DHTTYPE); // obiekt obsługujący czujnik DHT

#include <ESP_Mail_Client.h> // dołącz obsługę protokołu SMTP

// parametry konta, z którego będzie wysyłana poczta e-mail
SMTPSession smtp;

#define SMTP_HOST       "serwer poczty wychodzącej"
#define SMTP_PORT       numer portu SMTP (zazwyczaj 25, 465, 587, 2525)
#define SMTP_USER       "użytkownik"
#define SMTP_PASSWD     "hasło"
#define EMAIL_RECIPIENT "adres e-mail odbiorcy"

float alarmTemp = 25.0; // wartość krytyczna temperatury 
float alarmHmd = 75.0;  // i wilgotności 

int board_id = 1; // identyfikator urządzenia

const long eventEmail = 300000; // wysyłaj e-mail nie częściej niż co 5 minut
unsigned long previousEmail = eventEmail;

const long eventTime = 10000; // sprawdzaj dane z czujnika DHT co 10 sekund
unsigned long previousTime = eventTime;

void setup()
{
  Serial.begin(serialSpeed);

  Serial.print(F("Inicjalizacja czujnika DHT ... "));
  dht.begin();
  Serial.println(F("gotowe!"));

  // połącz z punktem dostępowym
  Serial.print(F("Łączę się z siecią WiFi "));
  WiFi.begin(wifi_ssid, wifi_passwd);

  while (WiFi.status() != WL_CONNECTED) { // czekaj na połączenie
    delay(500);
    Serial.print(F("."));
  }

  Serial.println(F(" gotowe!"));

  MailClient.networkReconnect(true); // jeśli konieczne to odzyskaj połączenie WiFi
  smtp.debug(0); // 1 - wysyłaj komunikaty techniczne na port szeregowy
  smtp.callback(smtpCallback); // status wysyłanych wiadomości

  Serial.print(F("Odczyt danych z czujnika DHT co "));
  Serial.print(eventTime / 1000);
  Serial.println(F(" sekund."));  
}

void loop() 
{
  unsigned long currentTime = millis();

  // obsługa kolejnego odczytu z czujnika DHT
  if (currentTime - previousTime >= eventTime)
  { 
    previousTime = currentTime;

    float temp = dht.readTemperature();
    float hmd = dht.readHumidity();

    if (!(isnan(temp) || isnan(hmd))) // czy odczyt z czujnika jest prawidłowy?
    {
      Serial.print(F("Odczyt danych z czujnika DHT => T: "));
      Serial.print(temp);
      Serial.print(F("°C, H: "));
      Serial.print(hmd);
      Serial.println(F("%"));

      // jeśli wartości krytyczne zostały przekroczone
      if (temp >= alarmTemp || hmd >= alarmHmd) {
        // wyślij e-mail, jeśli od ostatniej wysyłki minęło co najmniej 5 minut
        if (currentTime - previousEmail >= eventEmail)
        {
          previousEmail = currentTime;
          sendMessage(temp, hmd);
        }
      }
    }
    else 
    {
      Serial.println(F("Błędny odczyt z czujnika DHT."));
    }    
  }
}

void sendMessage(float temp, float hmd)
{
  ESP_Mail_Session session;

  // konfiguracja klienta SMTP
  session.server.host_name  = SMTP_HOST;
  session.server.port       = SMTP_PORT;
  session.login.email       = SMTP_USER;
  session.login.password    = SMTP_PASSWD;
  session.login.user_domain = "";

  // konfiguracja serwera NTP
  session.time.ntp_server = F("pool.ntp.org,time.nist.gov");
  session.time.gmt_offset = 2;
  session.time.day_light_offset = 1;

  SMTP_Message message;

  // tworzenie wiadomości e-mail
  message.sender.name = F("Domowy Monitoring");
  message.sender.email = SMTP_USER;
  message.subject = F("Alarm");
  message.addRecipient(F("Odbiorca"), EMAIL_RECIPIENT);

  String textMsg = F("Czujnik DHT [ID ");
  textMsg.concat(board_id);
  textMsg.concat(F("]:\nTemperatura: "));
  textMsg.concat(temp);
  textMsg.concat(F("°C\nWilgotność: "));
  textMsg.concat(String(hmd));
  textMsg.concat(F("%"));
  message.text.content = textMsg.c_str();
  message.text.charSet = F("utf-8");
  message.text.transfer_encoding = Content_Transfer_Encoding::enc_7bit;
  message.priority = esp_mail_smtp_priority::esp_mail_smtp_priority_high;
  
  // sprawdź połączenie z serwerem SMTP
  if (!smtp.connect(&session)) {
    return;
  }

  // wyślij wiadomość
  if (!MailClient.sendMail(&smtp, &message)) {
    Serial.println("Wystąpił problem z wysłaniem wiadomości e-mail, " + smtp.errorReason());
  }

  ESP_MAIL_PRINTF("Wolna pamięć: %d\n", MailClient.getFreeHeap());
}

void smtpCallback(SMTP_Status status)
{
  Serial.println(status.info()); // status wysłanej wiadomości

  // informacja o statusie wysłanej wiadomości
  if (status.success())
  {
    Serial.println("----------------");
    ESP_MAIL_PRINTF("Wiadomości wysłane poprawnie: %d\n", status.completedCount());
    ESP_MAIL_PRINTF("Wiadomości wysłane błędnie: %d\n", status.failedCount());
    Serial.println("----------------\n");
    struct tm dt;

    for (size_t i = 0; i < smtp.sendingResult.size(); i++)
    {
      // dane o wysłanej wiadomości
      SMTP_Result result = smtp.sendingResult.getItem(i);
      time_t ts = (time_t)result.timestamp;
      localtime_r(&ts, &dt);

      ESP_MAIL_PRINTF("Wiadomość nr: %d\n", i + 1);
      ESP_MAIL_PRINTF("Status: %s\n", result.completed ? "ok" : "błąd");
      ESP_MAIL_PRINTF("Date/Czas: %d/%d/%d %d:%d:%d\n", dt.tm_year + 1900, dt.tm_mon + 1, dt.tm_mday, dt.tm_hour, dt.tm_min, dt.tm_sec);
      ESP_MAIL_PRINTF("Odbiorca: %s\n", result.recipients.c_str());
      ESP_MAIL_PRINTF("Temat: %s\n", result.subject.c_str());
    }
    Serial.println("----------------\n");

    smtp.sendingResult.clear(); // zwolnij pamięć
  }
}
