// dołącz obsługę WiFi w zależności od zastosowanego układu ESP
#if defined(ESP8266)
  #include <ESP8266WiFi.h>
#elif defined(ESP32)
  #include <WiFi.h>
#endif

#define serialSpeed 115200     // prędkość transmisji UART
#define boardName "esp-device" // nazwa urządzenia IoT
#define boardID 1              // ID urządzenia IoT

char wifi_ssid[]   = "nazwa sieci WiFi";
char wifi_passwd[] = "hasło";

#include "DHT.h" // dołącz obsługę modułu DHT

/*
D2 - NodeMCU DevKit 0.9(V1)/1.0(V2)/V3 ESP-12/E
D8 - WeMos D1 R1/R2 (ESP8266)
5  - ESP-Mx DevKit (ESP8285)
8  - NodeMCU ESP-C3-12F
14 - WeMos D1 R32 (ESP-WROOM-32)
15 - NodeMCU-32-S2 ESP-12K, ESP-DevKit (ESP-WROOM-32)
*/

#define DHTPIN D2     // numer GPIO dla linii sygnałowej DAT
#define DHTTYPE DHT22 // typ czujnika DHT
/*
#define DHTTYPE DHT11
#define DHTTYPE DHT21 // DHT 21 (AM2301)
#define DHTTYPE DHT22 // DHT 22 (AM2302), AM2321
*/

DHT dht(DHTPIN, DHTTYPE); // obiekt obsługujący czujnik DHT

// dołącz obsługę klienta MQTT
#include "EspMQTTClient.h"

// konfiguracja dostępu do serwera MQTT
#define MQTTServer   "test.mosquitto.org" // bezpłatny serwer MQTT do testów
#define MQTTUsername "MQTTUsername"
#define MQTTPassword "MQTTPassword"
#define MQTTPort     1883
#define MQTTTopic    "SmartHome2022"

// definicja obiektu klienta MQTT
EspMQTTClient client(
  wifi_ssid,    // nazwa sieci WiFi
  wifi_passwd,  // hasło do sieci WiFi
  MQTTServer,   // adres lub IP serwera MQTT
  MQTTUsername, // nazwa użytkownika na serwerze MQTT (może być pusta)
  MQTTPassword, // hasło użytkownika na serwerze MQTT (może być puste)
  boardName,    // nazwa urządzenia IoT
  MQTTPort      // port serwera MQTT, wartość domyślna to 1883 (można pominąć)
);

int board_id = boardID; // identyfikator urządzenia

// subskrybowane tematy na serwerze MQTT
String mqtt_all = "";
String mqtt_board_info = "";
String mqtt_board_temp = "";
String mqtt_board_hmd = "";

const long eventTime = 10000; // sprawdzaj dane z czujnika DHT co 10 sekund
unsigned long previousTime = eventTime;

void setup()
{
  Serial.begin(serialSpeed);

  Serial.print(F("Inicjalizacja czujnika DHT ... "));
  dht.begin();
  Serial.println(F("gotowe!"));

  Serial.print(F("Wysyłka danych na serwer MQTT co "));
  Serial.print(eventTime / 1000);
  Serial.println(F(" sekund."));

  // definicja subskrybowanych tematów na serwerze MQTT
  mqtt_all = MQTTTopic;
  mqtt_all.concat(F("/#"));

  mqtt_board_info = MQTTTopic;
  mqtt_board_info.concat(F("/"));
  mqtt_board_info.concat(board_id);
  mqtt_board_info.concat(F("/info"));

  mqtt_board_temp = MQTTTopic;
  mqtt_board_temp.concat(F("/"));
  mqtt_board_temp.concat(board_id);
  mqtt_board_temp.concat(F("/temp"));

  mqtt_board_hmd = MQTTTopic;
  mqtt_board_hmd.concat(F("/"));
  mqtt_board_hmd.concat(board_id);
  mqtt_board_hmd.concat(F("/hmd"));

  // drukuj komunikaty techniczne na porcie szeregowym
  client.enableDebuggingMessages(); 

  // aktywuj możliwość aktualizacji firmware'u przez stronę WWW
  // adres strony http://deviceName.local
  // (login i hasło - MQTTUsername i MQTTPassword)
  // alternatywnie można użyc funkcji enableHTTPWebUpdater("user", "password")
  client.enableHTTPWebUpdater();
  
  // aktywuj możliwość aktualizacji przez OTA (ang. Over The Air)
  // hasło domyśłne to MQTTPassword
  // alternatywnie można użyc funkcji enableOTA("password", port)
  client.enableOTA();
}

// wykonaj polecenia po połaczeniu się z serwerem MQTT
void onConnectionEstablished()
{
  // zasubskrybuj temat SmartHome2022/1/temp z odczytami temperatury
  // komunikaty drukuj na porcie szeregowym
  client.subscribe(mqtt_board_temp, [](const String & content) {
    Serial.println(content);
  });

  // zasubskrybuj wszystkie tematy w SmartHome2022/#
  // komunikaty drukuj na porcie szeregowym
  client.subscribe(mqtt_all, [](const String & topic, const String & content) {
    Serial.println("(Wszystkie tematy) Temat: " + topic + ", Komunikat: " + content);
  });

  // opublikuj informację
  client.publish(mqtt_board_info, "Urządzenie zostało aktywowane!");

  // opublikuj informację z opóźnieniem 5 sekundowym
  client.executeDelayed(5 * 1000, []() {
    client.publish(mqtt_board_info, "Publikacja komunikatu po 5 sekundach.");
  });
}

void loop() 
{
  client.loop(); // obsługa klienta MQTT

  unsigned long currentTime = millis();

  // obsługa kolejnego odczytu z czujnika DHT
  if (currentTime - previousTime >= eventTime)
  { 
    previousTime = currentTime;

    float temp = dht.readTemperature();
    float hmd = dht.readHumidity();

    // jeśli odczyt z czujnika DHT jest prawidłowy, opublikuj dane na serwerze MQTT
    if (!(isnan(temp) || isnan(hmd)))
    {
      Serial.print(F("Odczyt danych z czujnika DHT => T: "));
      Serial.print(temp);
      Serial.print(F("°C, H: "));
      Serial.print(hmd);
      Serial.println(F("%"));

      // opublikuj wartość temperatury i wilgotności
      if (client.isConnected()) {
        client.publish(mqtt_board_temp, String(temp));
        client.publish(mqtt_board_hmd, String(hmd));
      }
    } 
    else 
    {
      Serial.println(F("Błędny odczyt z czujnika DHT."));
    }
  }
}
