<?php
namespace net\xmp\phpbook; 
 
class HTMLPage { 
 
    protected $title; 
    protected $content; 
    protected $template; 
 
    function __construct($title, $template='./template.html') { 
        $this->title = $title; 
        $this->content = ''; 
        $this->template = $template; 
    } 
 
    /** 
     * Wywietla utworzon stron HTML. 
     */ 
    function printPage() { 
        global $page; 
        $page = $this; 
        require $this->template; 
    } 
 
    /** 
     * Zwraca pojedyncze pole danych z obiektu HTMLPage. 
     * 
     * @param string $name Nazwa pola danych
     * @param boolean $escape Wskazuje, czy wynik powinien by maskowany 
     */ 
    function get($name, $escape=true) { 
        if ($escape) { 
            return $this->escape($this->$name); 
        } else { 
            return $this->$name; 
        } 
    } 
 
    /** 
     * Dodaje do strony element HTML zawierajcy tekst. 
     * 
     * @param string $name Nazwa elementu HTML
     * @param string $content Zawarto elementu 
     * @param boolean $keepWhiteSpace Pozostaw odstpy bez zmian
     */
    function addElement($name, $content, $keepWhiteSpace=false) { 
        $this->content .= "<$name>" . $this->escape($content, $keepWhiteSpace)  
                          . "</$name>"; 
    } 
 
    /** 
     * Dodaje do strony niefiltrowan zawarto HTML. 
     * 
     * @param string $html HTML od dodania 
     */ 
    function addHTML($html) { 
        $this->content .= $html; 
    } 
 
    /** 
     * Dodaje tabel do strony. 
     * 
     * @param array $content Tablica dwuwymiarowa: 
     *                       Pierwszy wiersz zawiera nagwek (<th>) 
     *                       Kolejne wiersze zawieraj dane (<td>) 
     * @param array $rawHTML Okrela, ktre kolumny s dodawane w postaci niefiltrowanego kodu HTML 
     */ 
    function addTable($content, $rawHTML=array()) { 
        $this->content .= "<table>\n"; 
        // Dodawanie nagwka
        $this->content .= "<thead><tr>\n"; 
        foreach ($content[0] as $cell) { 
            $this->addElement('th', $cell); 
        } 
        $this->content .= "</tr></thead>\n"; 
        // Dodawanie danych
        $this->content .= '<tbody>'; 
        for ($i = 1; $i < count($content); $i++) { 
            $this->content .= '<tr>'; 
            for ($j = 0; $j < count($content[$i]); $j++) { 
                if (empty($rawHTML[$j])) { 
                    $this->addElement('td', $content[$i][$j]); 
                } else { 
                    $this->addHTML('<td>' . $content[$i][$j] . '</td>'); 
                } 
            } 
            $this->content .= "</tr>\n"; 
        } 
        $this->content .= "</tbody></table>\n"; 
    } 
 
    /** 
     * Dodaje list do strony. 
     * 
     * @param array $items Lista pozycji. Jeeli pozycja jest tablic, to jej pierwszy element 
     *                     jest wyrniany, a drugi wywietlany normalnie 
     * @param boolean $rawHTML Wskazuje, czy dodawane pozycje s niefiltrowane.
     */
    function addList($items, $rawHTML=false) { 
        $this->content .= "<ul>\n"; 
        foreach ($items as $item) { 
            if (!is_array($item)) { 
                $rawHTML ? $this->addHTML('<li>' . $item . '</li>') 
                         : $this->addElement('li', $item); 
            } else { 
                $this->content .= '<li>'; 
                $this->addElement('b', $item[0] . ': '); 
                $this->content .= $rawHTML ? $item[1] : $this->escape($item[1]); 
                $this->content .= '</li>'; 
            } 
        } 
        $this->content .= "</ul>\n"; 
    } 
 
    /** 
     * Maskuje tekst wywietlany na stronie HTML. 
     * Zastpuje krytyczne znaki encjami i w zalenoci od parametru, usuwa 
     * wszystkie znaki o kodach mniejszych ni chr(32)
     * 
     * @param string $txt Tekst 
     * @param boolean $keepWhiteSpace Pozostawia odstpy bez zmian 
     */ 
    function escape($txt, $keepWhiteSpace=false) { 
        if ($keepWhiteSpace) { 
            $txt = preg_replace('/[\\x00-\\x08\\x0b-\\x1f]/', '', $txt); 
            return filter_var($txt, FILTER_SANITIZE_SPECIAL_CHARS); 
        } else { 
            $txt = preg_replace('/\s+/', ' ', $txt); 
            return filter_var($txt, FILTER_SANITIZE_SPECIAL_CHARS,  
                              FILTER_FLAG_STRIP_LOW); 
        } 
    }
}
