package com.packtpub.hibernatesearch.servlet;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.hibernate.Session;
import org.hibernate.search.FullTextQuery;
import org.hibernate.search.FullTextSession;
import org.hibernate.search.Search;
import org.hibernate.search.query.dsl.QueryBuilder;
import org.hibernate.search.query.facet.Facet;
import org.hibernate.search.query.facet.FacetSortOrder;
import org.hibernate.search.query.facet.FacetingRequest;
import org.hibernate.transform.AliasToBeanResultTransformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.packtpub.hibernatesearch.domain.App;
import com.packtpub.hibernatesearch.util.StartupDataLoader;

/**
 * Servlet implementujcy kontroler/model obsugujcy przeszukiwanie i renderowanie wynikw przy pomocy widoku w JSP/JSTL.
 * Adnotacja @Webservlet, dostpna w specyfikacji servletw w wersji 3.0, mapuje ten servlet do URLa "search (np. 
 * "http://localhost:8080/search"). We wczeniejszych wersjach specyfikacji ta konfiguracja znajdowaaby si w pliku "web.xml".
 * Podstawowa logika tej operacji wyszukiwania moe by zrefaktoryzowana w celu adaptacji do aplikacji napisanych w Springu, JSFie
 * lub dowolnym innym Javowym frameworku do aplikacji internetowych.
 */
@SuppressWarnings("serial")
@WebServlet("search")
public class SearchServlet extends HttpServlet {

	/**
	 * W tej metodzie zaimplementowane zostay gwne funkcjonalnoci wyszukiwania dla tego servletu. Metoda jest automatycznie wywoywana
	 * przy kadym wywoaniu dania HTTP POST do zmapowanego adresu URL.
	 */
	@SuppressWarnings("unchecked")
	@Override	
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		
		Logger logger = LoggerFactory.getLogger(SearchServlet.class);
		
		// Pobierz sowa kluczowe uytkownika.  Pobierz opcjonalne parametry stronicowania i sortowania lub uyj domylnych wartoci jeli nie podano.
		String searchString = request.getParameter("searchString") != null ? request.getParameter("searchString").trim() : "";
		String selectedDevice = request.getParameter("selectedDevice") != null ? request.getParameter("selectedDevice").trim() : "all";
		String selectedCategory = request.getParameter("selectedCategory") != null ? request.getParameter("selectedCategory") : "all";
		String selectedPriceRange = request.getParameter("selectedPriceRange") != null ? request.getParameter("selectedPriceRange") : "all";
		String sortField = request.getParameter("sortField") != null ? request.getParameter("sortField").trim() : "relevance";
		int firstResult = request.getParameter("firstResult") != null ? Integer.parseInt(request.getParameter("firstResult")) : 0;
		logger.info("Zapytanie [" + searchString 
				+ "], urzdzenie [" + selectedDevice
				+ "], kategoria [" + selectedCategory
				+ "], zakres cen [" + selectedPriceRange
				+ "], sortowanie [" + sortField 
				+ "], zacznij od wyniku [" +  firstResult + "]");

		// Rozpocznij sesj Hibernate.
		Session session = StartupDataLoader.openSession();
		
		// Utwrz wrapper Hibernate Search wok czystej sesji Hibernate
		FullTextSession fullTextSession = Search.getFullTextSession(session);

		// Rozpocznij transkacj. W tym przypadku nie jest to niezbdne, ale jest tzw. dobr praktyk.
		fullTextSession.beginTransaction();

		// Utwrz obiekt typu QueryBuilder pochodzcy z Hibernate Search i utwrz go dla odpowiedniego indeksu Lucene (w naszym przypadku indeksu "App").
		QueryBuilder queryBuilder = fullTextSession.getSearchFactory().buildQueryBuilder().forEntity( App.class ).get();
		
		// Uyj QueryBuildera by zbudowa w Lucene zapytanie z uyciem sw kluczowych, porwnujce sowa kluczowe wprowadzone przez uytkownika z polami 
		// name i description w klasie App, pola name powizanych aplikacji oraz pola comments wbudowanych obiektw klasy CustomerReview.
		org.apache.lucene.search.Query luceneQuery = null;
		if(searchString.length() > 2 && searchString.startsWith("\"") && searchString.endsWith("\"")) {
			
			// Jeeli sowa kluczowe wprowadzone przez uytkownika s umieszczone w cudzysowach, uyj wyszukiwania z uyciem frazy
			// Zwiksz wag pl "name" oraz "description" 
			String unquotedSearchString = searchString.substring(1, searchString.length() - 1);
			luceneQuery = queryBuilder
					.phrase()
					.onField("name").boostedTo(2)
					.andField("description").boostedTo(2)
					.andField("supportedDevices.name").andField("customerReviews.comments")
					.sentence(unquotedSearchString)
					.createQuery();
		} else {
			
			// Jeeli sowa kluczowe wprowadzone przez uytkownika nie s umieszczone w cudzysowach, uyj rozmytego wyszukiwania z uyciem sw kluczowych
			luceneQuery = queryBuilder
					.keyword()
					.fuzzy()
					.withThreshold(0.7f)
					.onFields("name", "description", "supportedDevices.name", "customerReviews.comments")
					.matching(searchString)
					.createQuery();
		}
		FullTextQuery hibernateQuery = fullTextSession.createFullTextQuery(luceneQuery, App.class);  // moe by rzutowane do "org.hibernate.Query"

		// Uyj FullTextFilter aby wyszuka tylko aplikacje wspierane przez konkretne urzdzenie 
		if(selectedDevice != null && !selectedDevice.equals("all")) {
			hibernateQuery.enableFullTextFilter("deviceName").setParameter("deviceName", selectedDevice);
		}
		
		// Dodaj opcjonalne kryteria sortowania, jeeli nie zostay ustawione, sortuj wg wanoci
		if(sortField.equals("name")) {
			Sort sort = new Sort(new SortField("sorting_name", SortField.STRING));
			hibernateQuery.setSort(sort);
		} else if(sortField.equals("name-reverse")) {
			Sort sort = new Sort(new SortField("sorting_name", SortField.STRING, true));
			hibernateQuery.setSort(sort);
		}

		// Pobierz szacowan liczb wynikw (UWAGA: nie jest 100% dokadna, ale nie wymaga odpytania bazy danych 
		// co moe by kosztowne przy duych zbiorach danych).
		int resultSize = hibernateQuery.getResultSize();

		// Wykonaj wyszukiwanie ograniczone do 5 wynikw, oraz punktem pocztkowym w miejscu gdzie poprzednia strona zakoczya
		// prezentacj wynikw (domylnie od pocztku)
		logger.info("Zapytanie == " + hibernateQuery.getQueryString());
		hibernateQuery.setFirstResult(firstResult);
		hibernateQuery.setMaxResults(5);
		
		// Uyj projekcji by pobra dane z indeksu Lucene zamiast z bazy danych.  Nie potrzebujemy wszystkich pl encji App, 
		// wycznie te ktre chcemy wywietla oraz "id" ebymy mogli pobra obiekt jeeli bdzie potrzebny. 
		hibernateQuery.setProjection("id", "name", "description", "image");
		
		// Poniewa uywamy projekcji, "hibernateQuery.list()" standardowo zwrci "Object[]" (tj. po jednej tablicy 
		// na kade pole).  ResultTransformer uyje tych wartoci by zbudowa obiekty typu App i sprawi, e "hibernateQuery.list()"
		// zwrci "List<App>".
		hibernateQuery.setResultTransformer( new AliasToBeanResultTransformer(App.class) );		
		
		// utwrz dyskretne danie fasetowe, by okreli jakie kategorie znajduj si w wynikach wyszukiwania 
		// oraz ile wynikw znajduje si w kadej kategorii. Te informacje bd dostpne dopiero po zakoczeniu 
		// zapytania (tzn. po wywoaniu "hibernateQuery.list()").
		FacetingRequest categoryFacetingRequest = queryBuilder
				.facet()
				.name("categoryFacet")
				.onField("category")
				.discrete()
				.orderedBy(FacetSortOrder.FIELD_VALUE)
				.includeZeroCounts(false)
				.createFacetingRequest();
		hibernateQuery.getFacetManager().enableFaceting(categoryFacetingRequest);
		
		// Utwrz danie z fasetem opartym o kategorie, by okreli ktre wyniki wyszukiwania mieszcz si w zakresach cen "poniej 1$", "1$-5$", 
		// i "powyej 5$". Te informacje bd dostpne dopiero po zakoczeniu zapytania (tzn. po wywoaniu "hibernateQuery.list()").
		FacetingRequest priceRangeFacetingRequest = queryBuilder
				.facet()
				.name("priceRangeFacet")
				.onField("price")
				.range()
				.below(1f).excludeLimit()
				.from(1f).to(5f)
				.above(5f).excludeLimit()
				.createFacetingRequest();
		hibernateQuery.getFacetManager().enableFaceting(priceRangeFacetingRequest);
		
		// Jeeli zapytanie nie zakoczy si przed upywem dwch sekund, zwrc pobrane do tej pory wyniki.
		hibernateQuery.limitExecutionTimeTo(2, TimeUnit.SECONDS);
		
		// Wykonaj zapytanie
		List<App> apps = hibernateQuery.list();
		
		// Po wykonaniu zapytania pobierz dane do fasetw zdefiniowanych powyej 
		List<Facet> categoryFacets = hibernateQuery.getFacetManager().getFacets("categoryFacet");
		List<Facet> priceRangeFacets = hibernateQuery.getFacetManager().getFacets("priceRangeFacet");
		
		// Uyj fasetu z kategoriami aby zbudowa list kategorii obecnych w tym zbiorze danych.
		Map<String, Integer> categories = new TreeMap<String, Integer>();
		for(Facet categoryFacet : categoryFacets) {
			categories.put(categoryFacet.getValue(), categoryFacet.getCount());
			// Jeeli do servletu zosta przekazany parametr "selectedCategory", oznacza to e dana kategoria zostaa
			// wybrana na stronie z wynikami. Dodaj pasujcy faset do zapytania i przeprowad je ponownie wycznie 
			// dla tej kategorii. 
			if(categoryFacet.getValue().equalsIgnoreCase(selectedCategory)) {
				hibernateQuery.getFacetManager().getFacetGroup("categoryFacet").selectFacets(categoryFacet);
			}
		}
		// Uyj fasetu z zakresami cen aby zbudowa list zakresw cen wynikw.
		Map<String, Integer> priceRanges = new TreeMap<String, Integer>();
		for(Facet priceRangeFacet : priceRangeFacets) {
			priceRanges.put(priceRangeFacet.getValue(), priceRangeFacet.getCount());
			// Jeeli do servletu zosta przekazany parametr "selectedPriceRange", oznacza to e dana kategoria zostaa
			// wybrana na stronie z wynikami. Dodaj pasujcy faset do zapytania i przeprowad je ponownie wycznie 
			// dla tej kategorii. 
			if(priceRangeFacet.getValue().equalsIgnoreCase(selectedPriceRange)) {
				hibernateQuery.getFacetManager().getFacetGroup("priceRangeFacet").selectFacets(priceRangeFacet);
			}
		}
		// Wykonaj ponownie zapytanie. Jeeli aden faset nie zosta dodany, wyniki nie zmieni si.
		apps = hibernateQuery.list();
		
		// Pobierz szacowan liczb wynikw aby umozliwi stronicowanie wynikw.
		resultSize = hibernateQuery.getResultSize();
		
		// Umie wyniki wyszukiwania w obiekcie dania HTTP, wraz z parametrami sortowania i stronicowania
		request.setAttribute("searchString", searchString);
		request.setAttribute("selectedDevice", selectedDevice);
		request.setAttribute("selectedCategory", selectedCategory);
		request.setAttribute("selectedPriceRange", selectedPriceRange);
		request.setAttribute("sortField", sortField);
		request.setAttribute("apps", apps);
		request.setAttribute("resultSize", resultSize);
		request.setAttribute("firstResult", firstResult);
		request.setAttribute("categories", categories);
		request.setAttribute("priceRanges", priceRanges);

		// Oprnij i zamknij sesj Hibernate.
		fullTextSession.getTransaction().commit();
		session.close();
		
		// Przeka danie HTTP, wraz z wynikami wyszukiwania, do widoku zbudowanego JSP/JSTL w celu wyrenderowania strony z wynikami
		getServletContext().getRequestDispatcher("/WEB-INF/pages/search.jsp").forward(request, response);
	}

	/**
	 * Ta metoda jest wywoywana za kadym razem gdy zostanie wywoane danie HTTP GET do zmapowanego URLa.  Dla naszego servletu nie ma znaczenia, 
	 * czy zostanie wywoany metod POST czy GET. W zwizku z tym przekierowujemy danie do metody "doPost()".
	 */
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {	
		this.doPost(request, response);
	}
	
}
