package com.packtpub.hibernatesearch.domain;

import javax.persistence.Embeddable;

import org.apache.solr.analysis.HTMLStripCharFilterFactory;
import org.apache.solr.analysis.StandardTokenizerFactory;
import org.apache.solr.analysis.StandardFilterFactory;
import org.apache.solr.analysis.StopFilterFactory;
import org.hibernate.search.annotations.Analyzer;
import org.hibernate.search.annotations.AnalyzerDef;
import org.hibernate.search.annotations.CharFilterDef;
import org.hibernate.search.annotations.DynamicBoost;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.NumericField;
import org.hibernate.search.annotations.TokenFilterDef;
import org.hibernate.search.annotations.TokenizerDef;

import com.packtpub.hibernatesearch.util.FiveStarBoostStrategy;

/**
 * Osadzalny obiekt (lub "komponent" w tradycyjnym argonie Hibernate), przechowujcy opini uytkownika o aplikacji. 
 * Opinia skada si z nazwy uytkownika prezentujcego opini, liczby gwiazdek (maksimum pi) oznaczajcych ocen aplikacji,
 * oraz komentarza.
 * 
 * Adnotacja @Embeddable informuje Hibernate, e czas ycia obiektu jest zaleny od czasu ycia obiektu w ktrym si zawiera.
 * Innymi sowy, gdy aplikacja zostanie skasowana, razem z ni zostan skasowane komentarze jej dotyczce.
 *
 * Poniewa ta klasa nie jest adnotowana za pomoc @Indexed, Hibernate Search nie utworzy dla niej indeksu Lucene.
 * Wszystkie informacje wskazane do indeksowania trafi do indeksu Lucene obiektu w ktrym si zawiera. 
 * 
 * Adnotacja @AnalyzerDef definiuje powizanie analyzera z polem "comments". Analyzer usuwa tagi HTML oraz
 * dodaje kilka filtrw tokenw usuwajcych literwki.
 *
 * Adnotacja @DynamicBoost dynamicznie ustawia wag pola dla obiektw tej klasy w trakcie pracy aplikacji uywajc klasy 
 * FiveStarBoostStrategy.
 */
@Embeddable
@AnalyzerDef(
	name="customerReviewAnalyzer",
	charFilters={ @CharFilterDef(factory=HTMLStripCharFilterFactory.class) },
	tokenizer=@TokenizerDef(factory=StandardTokenizerFactory.class),
	filters={ 
		@TokenFilterDef(factory=StandardFilterFactory.class),
		@TokenFilterDef(factory=StopFilterFactory.class)
	}
)
@DynamicBoost(impl=FiveStarBoostStrategy.class)
public class CustomerReview {

	/**
	 * Nazwa uytkownika osoby wysyajcej komentarz. Adnotacja @Column jest zbdna dla osadzalnych obiektw, 
	 * jednak adnotacja Hibernate Search @Field jest nadal uywana by zmapowa pole na indeks Lucene dowolnej encji, w ktrej 
	 * obiekt zosta osadzony (np. App).
	 */	
	@Field
	private String username;
	
	/**
	 * Ocena w skali od 1 do 5 (np. "4 z 5 gwiazdek").  Adnotacja @NumericField instruuje Hibernate Search by indeksowa pole uywajc
	 * wyspecjalizowanej struktury danych, by usprawni sortowanie oraz zapytania oparte o zakres.
	 */
	@Field
	@NumericField
	private int stars;
	
	/**
	 * Komentarz dotyczcy aplikacji.  Adnotacja @Column jest zbdna dla osadzalnych obiektw, 
	 * jednak adnotacja Hibernate Search @Field jest nadal uywana by zmapowa pole na indeks Lucene dowolnej encji, w ktrej 
	 * obiekt zosta osadzony (np. App).
	 * 
	 * Adnotacja @Analyzer czy zdefiniowany analyzer z polem.
	 */		
	@Field
	@Analyzer(definition="customerReviewAnalyzer")
	private String comments;

	/**
	 * Default empty constructor.
	 */
	public CustomerReview() {
	}
	
	/**
	 * Wygodny konstruktor, ustawiajcy wszystkie wartoci pl jednoczenie.  
	 */
	public CustomerReview(String username, int stars, String comments) {
		this.username = username;
		this.stars = stars;
		this.comments = comments;
	}

	//
	// GETTERY I SETTERY
	//
	
	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public int getStars() {
		return stars;
	}

	public void setStars(int stars) {
		this.stars = stars;
	}

	public String getComments() {
		return comments;
	}

	public void setComments(String comments) {
		this.comments = comments;
	}
	
}
