package com.packtpub.hibernatesearch.domain;

import java.io.Serializable;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.ManyToMany;

import org.hibernate.annotations.Fetch;
import org.hibernate.annotations.FetchMode;
import org.hibernate.search.annotations.Analyze;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Fields;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.IndexedEmbedded;

/**
 * Prosta klasa encji reprezentujca aplikacj komputerow ... z nazw, dugim opisem 
 * i nazw pliku uytego jako ikona. Klasa posiada flag "active" okrelajc, czy dana aplikacja
 * ma by wyszukiwalna.
 * 
 * Adnotacja @Entity nakazuje Hibernate by zmapowao klas do tabeli w bazie danych, podczas gdy adnotacja @Indexed 
 * nakazuje narzdziu Hibernate Search by zmapowao j do indeksu Lucene. 
 */
@Entity
@Indexed
public class App implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * Klucz gwny, skonfigurowany tak, by by generowany automatycznie podczas tworzenia nowej instancji
	 */
	@Id
	@GeneratedValue
	private Long id;

	/**
	 * Nazwa aplikacji zrozumiaa dla ludzi. Adnotacja @Column informuje Hibernate by zmapowa to pole do 
	 * kolumny w bazie danych.
	 * 
	 * Obecnie uywamy dwch adnotacji @Field w celu dwukrotnego indeksowania... Pierwsze z przetworzonymi (rozbitymi) 
	 * wartociami dla atwiejszego wyszukiwania, drugie nieprzetworzone (nie rozbite) aby na podstawie ich wartoci 
	 * sortowa wyniki.
	 */
	@Column
	@Fields({
		@Field,
		@Field(name="sorting_name", analyze=Analyze.NO)
	})
	private String name;

	/**
	 * Duszy, bardziej szczegowy opis aplikacji. Adnotacja @Column informuje Hibernate by zmapowa to pole do 
	 * kolumny w bazie danych, natomiast adnotacja @Field nakazuje Hibernate Search by zmapowa je jako pole  
	 * indeksie Lucene.
	 */
	@Column(length = 1000)
	@Field
	private String description;

	/**
	 * Nazwa pliku obrazka majcego by powizanym z aplikacj. Spodziewamy si, e plik bdzie istnia w podkatalogu 
	 * "images/apps/<obrazek>".  To pole nie ma adnotacji @Field, poniewa nie planujemy przeszukiwania aplikacji pod  
	 * ktem nazw plikw obrazkw.
	 */
	@Column
	private String image;
	
	/**
	 * Flaga okrelajca, czy aplikacja ma by wyszukiwalna.
	 */
	@Column
	private boolean active;

	/**
	 * Kolekcja powizanych encji typu Device, reprezentujcych urzdzenia na ktrych mona uruchomi aplikacj.
	 * 
	 * Adnotacja @ManyToMany informuje Hibernate, e aplikacja moe by uruchamiana na wielu urzdzeniach, oraz e
	 * jedno urzdzenie moe uruchamia wiele aplikacji  
	 * 
	 *      Element "fetch" ma ustawion warto "eager" (zamiast standardowej "lazy") aby wszystkie interesujce nas obiekty typu 
	 *      Device zostay pobrane jednoczenie dopki sesja Hibernate jest otwarta. W innym przypadku pojawiyby si bdy gdyby 
	 *      warstwa widoku sprbowaa dosta si do tych pl po zamkniciu sesji Hibernate. Zazwyczaj chciwe pobieranie jest mniej  
	 *      wydajne od leniwego. W wikszej aplikacji powiniene si zastanowi nad innym podejciem, np. uyciem wzorca DAO. 
	 * 
	 *      Element "cascade" zapewnia, e zmiany po dowolnej stronie relacji App-Device bd poprawnie odzwierciedlone w indeksach 
	 *      Lucene obu encji.
	 * 
	 * Adnotacja @IndexedEmbedded informuje Hibernate Search by indeksowa powizane encje.  
	 * Klasa Device moe (ale nie musi) posiada indeks Lucene (w zalenoci od tego czy klasa jest adnotowana @Indexed), 
	 * ale i tak instancje Device powizane z aplikacj bd przechowywane w indeksie Lucene klasy App. 
	 */
	@ManyToMany(fetch=FetchType.EAGER, cascade = { CascadeType.ALL })
	@IndexedEmbedded(depth=1)
	private Set<Device> supportedDevices;
	
	/**
	 * Kolekcja wbudowanych obiektw prezentujcych opinie uytkownikw o aplikacji. Wbudowane obiekty rni si od powizanych encji 
	 * tym, e z chwil skasowania obiektu, z ktrym s powizane one rwnie zostaj usunite. 
	 * 
	 * Adnotacja @ElementCollection informuje Hibernate o powizaniu z wieloma obiektami wbudowanymi, w podobny sposb co adnotacja 
	 * @ManyToMany uyta dla "supportedDevices".  Element "fetch" ma ustawion warto "Eager" z tego samego powodu co w przypadku 
	 * "SupportedDevices".   
	 *
	 * Adnotacja @Fetch instruuje Hibernate by pobrao wbudowane obiekty uywajc wielu polece SELECT zamiast jednego rozbudowanego 
	 * JOIN (domylnego zachowania).  Takie podejcie jest niewydajne dla duych zbiorw danych ... ale spodziewamy si, e pojedycza 
	 * aplikacja nie bdzie miaa zbyt wielu komentarzy. Dziki temu podejciu unikniemy zwrcenia przez Hibernate zduplikowanych 
	 * komentarzy (http://stackoverflow.com/questions/1093153/hibernate-collectionofelements-eager-fetch-duplicates-elements). Jak  
	 * wspomniano w komentarzu do "supportedDevices", w wikszych aplikacjach zastanw si nad refaktoringiem rozwizania tak, aby 
	 * unikn zachannego pobierania obiektw.
	 * 
	 * Adnotacja @IndexEmbedded ma zdefiniowany opcjonalny element "includePaths" by wskaza, ktre pola wbudowanych obiektw 
	 * bd indeksowane w encji nadrzdnej.  Innymi sowy pole "comments" bdzie jedyn wartoci klasy CustomerReview doczon
	 * do indeksu Lucene klasy App.    
	 */
	@ElementCollection(fetch=FetchType.EAGER)
	@Fetch(FetchMode.SELECT)
	@IndexedEmbedded(depth=1, includePaths = { "comments" })
	private Set<CustomerReview> customerReviews;

	/**
	 * Domylny pusty konstruktor.
	 */
	public App() {
	}

	/**
	 * Wygodny konstruktor, ustawiajcy zawarto wszystkich pl w jednym kroku.  
	 * 
	 * Parametr "active" jest domylnie ustawiony na warto "true".  Aby dezaktywowa aplikacj,
	 * uyj settera dla tego pola. 
	 */
	public App(String name, String image, String description) {
		this.name = name;
		this.image = image;
		this.description = description;
		this.active = true;
	}

	//
	// GETTERY I SETTERY
	//
	
	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getImage() {
		return image;
	}

	public void setImage(String image) {
		this.image = image;
	}

	public Set<Device> getSupportedDevices() {
		return supportedDevices;
	}

	public void setSupportedDevices(Set<Device> supportedDevices) {
		this.supportedDevices = supportedDevices;
	}

	public boolean isActive() {
		return active;
	}

	public void setActive(boolean active) {
		this.active = active;
	}

	public Set<CustomerReview> getCustomerReviews() {
		return customerReviews;
	}

	public void setCustomerReviews(Set<CustomerReview> customerReviews) {
		this.customerReviews = customerReviews;
	}

}
