package com.packtpub.hibernatesearch.domain;

import java.io.Serializable;
import java.util.Date;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.ManyToMany;

import org.apache.solr.analysis.HTMLStripCharFilterFactory;
import org.apache.solr.analysis.PhoneticFilterFactory;
import org.apache.solr.analysis.SnowballPorterFilterFactory;
import org.apache.solr.analysis.StandardFilterFactory;
import org.apache.solr.analysis.StandardTokenizerFactory;
import org.apache.solr.analysis.StopFilterFactory;
import org.hibernate.annotations.Fetch;
import org.hibernate.annotations.FetchMode;
import org.hibernate.search.annotations.Analyze;
import org.hibernate.search.annotations.Analyzer;
import org.hibernate.search.annotations.AnalyzerDef;
import org.hibernate.search.annotations.CharFilterDef;
import org.hibernate.search.annotations.Boost;
import org.hibernate.search.annotations.DateBridge;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Fields;
import org.hibernate.search.annotations.FullTextFilterDefs;
import org.hibernate.search.annotations.FullTextFilterDef;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.IndexedEmbedded;
import org.hibernate.search.annotations.NumericField;
import org.hibernate.search.annotations.Parameter;
import org.hibernate.search.annotations.Resolution;
import org.hibernate.search.annotations.Store;
import org.hibernate.search.annotations.TokenFilterDef;
import org.hibernate.search.annotations.TokenizerDef;

import com.packtpub.hibernatesearch.util.DeviceFilterFactory;
import com.packtpub.hibernatesearch.util.IndexWhenActiveInterceptor;

/**
 * Prosta klasa encji reprezentujca aplikacj komputerow ... z nazw, dugim opisem 
 * i nazw pliku uytego jako ikona. Klasa posiada flag "active" okrelajc, czy dana aplikacja
 * ma by wyszukiwalna.  
 * 
 * Adnotacja @Entity nakazuje Hibernate by zmapowao klas do tabeli w bazie danych, podczas gdy adnotacja @Indexed 
 * nakazuje narzdziu Hibernate Search by zmapowao j do indeksu Lucene. Element "interceptor" ustawia warunki 
 * indeksowania by zapobiec indeksowaniu instancji zawierajcych flagi "active" ustawione na "false" (patrz klasa 
 * "com.packtpub.hibernatesearch.util.IndexWhenActiveInterceptor").
 * 
 * Adnotacja @AnalyzerDef definiuje powizanie analyzera z polem "description". Analyzer usuwa tagi HTML oraz
 * dodaje kilka filtrw tokenw usuwajcych literwki.
 */
@Entity
@Indexed(interceptor=IndexWhenActiveInterceptor.class)
@AnalyzerDef(
	name="appAnalyzer",
	charFilters={ @CharFilterDef(factory=HTMLStripCharFilterFactory.class) },
	tokenizer=@TokenizerDef(factory=StandardTokenizerFactory.class),
	filters={ 
		@TokenFilterDef(factory=StandardFilterFactory.class),
		@TokenFilterDef(factory=StopFilterFactory.class),
		@TokenFilterDef(factory=PhoneticFilterFactory.class, params = {
        	@Parameter(name="encoder", value="DoubleMetaphone")
		}),
		@TokenFilterDef(factory=SnowballPorterFilterFactory.class, params = {
			@Parameter(name="language", value="English") 
		})
	}
)
@Boost(2.0f)
@FullTextFilterDefs({
	@FullTextFilterDef(name="deviceName", impl=DeviceFilterFactory.class)
})
public class App implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * Klucz gwny, skonfigurowany tak, by by generowany automatycznie podczas tworzenia nowej instancji
	 */
	@Id
	@GeneratedValue
	private Long id;

	/**
	 * Nazwa aplikacji zrozumiaa dla ludzi. Adnotacja @Column informuje Hibernate by zmapowa to pole do 
	 * kolumny w bazie danych.
	 * 
	 * Obecnie uywamy dwch adnotacji @Field w celu dwukrotnego indeksowania... Pierwsze z przetworzonymi (rozbitymi) 
	 * wartociami dla atwiejszego wyszukiwania, drugie nieprzetworzone (nie rozbite) aby na podstawie ich wartoci 
	 * sortowa wyniki.
	 * 
	 * Adnotacja @Boost uyta na poziomie pola nadaje mu jeszcze wiksz wag podczas wyliczania wanoci wynikw wyszukiwania.
	 * Warto 1.5 kumuluje si z wartoci 2.0 ustawion na klasie.
	 */
	@Column
	@Fields({
		@Field(store=Store.COMPRESS),
		@Field(name="sorting_name", analyze=Analyze.NO)
	})
	@Boost(1.5f)
	private String name;

	/**
	 * Duszy, bardziej szczegowy opis aplikacji. Adnotacja @Column informuje Hibernate by zmapowa to pole do 
	 * kolumny w bazie danych, natomiast adnotacja @Field nakazuje Hibernate Search by zmapowa je jako pole  
	 * indeksie Lucene.
	 * 
	 * Adnotacja @Analyzer wie pole ze zdefiniowanym powyej analyzerem.
	 * 
	 * Adnotacja @Boost uyta na poziomie pola nadaje mu jeszcze wiksz wag podczas wyliczania wanoci wynikw wyszukiwania. 
	 * Warto 1.2 kumuluje si z wartoci 2.0 ustawion na klasie.
	 */
	@Column(length = 1000)
	@Field(store=Store.COMPRESS)
	@Analyzer(definition="appAnalyzer")
	@Boost(1.2f)
	private String description;
	
	@Column
	@Field
	private String category;

	/**
	 * Nazwa pliku obrazka majcego by powizanym z aplikacj. Spodziewamy si, e plik bdzie istnia w podkatalogu 
	 * "images/apps/<obrazek>".  To pole nie ma adnotacji @Field, poniewa nie planujemy przeszukiwania aplikacji pod  
	 * ktem nazw plikw obrazkw.
	 */
	@Column
	@Field(store=Store.COMPRESS)
	private String image;
	
	/**
	 * Cena aplikacji. Adnotacja @NumericField instruuje Hibernate Search by indeksowa pole uywajc
	 * wyspecjalizowanej struktury danych, by usprawni sortowanie oraz zapytania oparte o zakres.
	 */
	@Column
	@Field
	@NumericField
	private float price;
	
	/**
	 * Flaga okrelajca, czy aplikacja ma by wyszukiwalna.
	 */
	@Column
	private boolean active;

	/**
	 * Data wydania aplikacji.  Opcjonalna adnotacja @DateBridge uywa elementu "resolution" by zadeklarowa,
	 * e data ma by indeksowana do poziomu dnia... zamiast standardowego podejcia tj. zapisywania 
	 * penej daty, co do milisekundy.
	 */
	@Column
	@Field
	@DateBridge(resolution=Resolution.DAY)
	private Date releaseDate;

	/**
	 * Kolekcja powizanych encji typu Device, reprezentujcych urzdzenia na ktrych mona uruchomi aplikacj.
	 * 
	 * Adnotacja @ManyToMany informuje Hibernate, e aplikacja moe by uruchamiana na wielu urzdzeniach, oraz e
	 * jedno urzdzenie moe uruchamia wiele aplikacji  
	 * 
	 *      Element "fetch" ma ustawion warto "eager" (zamiast standardowej "lazy") aby wszystkie interesujce nas obiekty typu 
	 *      Device zostay pobrane jednoczenie dopki sesja Hibernate jest otwarta. W innym przypadku pojawiyby si bdy gdyby 
	 *      warstwa widoku sprbowaa dosta si do tych pl po zamkniciu sesji Hibernate. Zazwyczaj chciwe pobieranie jest mniej  
	 *      wydajne od leniwego. W wikszej aplikacji powiniene si zastanowi nad innym podejciem, np. uyciem wzorca DAO. 
	 * 
	 *      Element "cascade" zapewnia, e zmiany po dowolnej stronie relacji App-Device bd poprawnie odzwierciedlone w indeksach 
	 *      Lucene obu encji.
	 * 
	 * Adnotacja @IndexedEmbedded informuje Hibernate Search by indeksowa powizane encje.  
	 * Klasa Device moe (ale nie musi) posiada indeks Lucene (w zalenoci od tego czy klasa jest adnotowana @Indexed), 
	 * ale i tak instancje Device powizane z aplikacj bd przechowywane w indeksie Lucene klasy App.
	 */
	@ManyToMany(fetch=FetchType.EAGER, cascade = { CascadeType.ALL })
	@IndexedEmbedded(depth=1)
	private Set<Device> supportedDevices;
	
	/**
	 * Kolekcja wbudowanych obiektw prezentujcych opinie uytkownikw o aplikacji. Wbudowane obiekty rni si od powizanych encji 
	 * tym, e z chwil skasowania obiektu, z ktrym s powizane one rwnie zostaj usunite. 
	 * 
	 * Adnotacja @ElementCollection informuje Hibernate o powizaniu z wieloma obiektami wbudowanymi, w podobny sposb co adnotacja 
	 * @ManyToMany uyta dla "supportedDevices".  Element "fetch" ma ustawion warto "Eager" z tego samego powodu co w przypadku 
	 * "SupportedDevices".   
	 *
	 * Adnotacja @Fetch instruuje Hibernate by pobrao wbudowane obiekty uywajc wielu polece SELECT zamiast jednego rozbudowanego 
	 * JOIN (domylnego zachowania).  Takie podejcie jest niewydajne dla duych zbiorw danych ... ale spodziewamy si, e pojedycza 
	 * aplikacja nie bdzie miaa zbyt wielu komentarzy. Dziki temu podejciu unikniemy zwrcenia przez Hibernate zduplikowanych 
	 * komentarzy (http://stackoverflow.com/questions/1093153/hibernate-collectionofelements-eager-fetch-duplicates-elements). Jak  
	 * wspomniano w komentarzu do "supportedDevices", w wikszych aplikacjach zastanw si nad refaktoringiem rozwizania tak, aby 
	 * unikn zachannego pobierania obiektw.
	 */
	@ElementCollection(fetch=FetchType.EAGER)
	@Fetch(FetchMode.SELECT)
	@IndexedEmbedded(depth=1, includePaths = { "stars", "comments" })
	private Set<CustomerReview> customerReviews;

	/**
	 * Domylny pusty konstruktor.
	 */
	public App() {
	}

	
	/**
	 * Wygodny konstruktor, ustawiajcy zawarto wszystkich pl w jednym kroku.  
	 * 
	 * Parametr "active" jest domylnie ustawiony na warto "true".  Aby dezaktywowa aplikacj,
	 * uyj settera dla tego pola. 
	 */
	public App(String name, String image, String description, String category, float price) {
		this.name = name;
		this.image = image;
		this.description = description;
		this.category = category;
		this.price = price;
		
		// Pozostae pola inicjujemy z sensownymi wartociami domylnymi. StartupDataLoader moe je 
		// zmieni setterami
		this.active = true;
		this.releaseDate = new Date();
	}

	//
	// GETTERY I SETTERY
	//
	
	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getCategory() {
		return category;
	}

	public void setCategory(String category) {
		this.category = category;
	}

	public String getImage() {
		return image;
	}

	public void setImage(String image) {
		this.image = image;
	}

	public Set<Device> getSupportedDevices() {
		return supportedDevices;
	}

	public void setSupportedDevices(Set<Device> supportedDevices) {
		this.supportedDevices = supportedDevices;
	}

	public boolean isActive() {
		return active;
	}

	public void setActive(boolean active) {
		this.active = active;
	}

	public Set<CustomerReview> getCustomerReviews() {
		return customerReviews;
	}

	public void setCustomerReviews(Set<CustomerReview> customerReviews) {
		this.customerReviews = customerReviews;
	}

	public Date getReleaseDate() {
		return releaseDate;
	}

	public void setReleaseDate(Date releaseDate) {
		this.releaseDate = releaseDate;
	}

	public float getPrice() {
		return price;
	}

	public void setPrice(float price) {
		this.price = price;
	}

}
