package com.packtpub.hibernatesearch.util;

import org.hibernate.search.indexes.interceptor.EntityIndexingInterceptor;
import org.hibernate.search.indexes.interceptor.IndexingOverride;

import com.packtpub.hibernatesearch.domain.App;

/**
 * EntityIndexingInterceptor dla encji App, uywany do warunkowego indeksowania. Interceptor ma za zadanie 
 * zapobieeniu indeksowania nieaktywnych aplikacji (tzn. instancje App, ktrych pole "active" ma warto 
 * "false").  
 * 
 * Adnotacja @Indexed klasy encji App rejestruje interceptor i sprawia, e jedna z jej metod jest uruchamiana 
 * gdy wykonywane s jakie operacje na instancji App na poziomie Hibernate.  Odpowiednia metoda okrela co 
 * powinno si zdarzy z indeksem Lucene na poziomie Hibernate Search. 
 * 
 * UWAGA: Warunkowe indeksowanie nie jest niezawodne, poniewa interceptory s ignorowane, gdy Lucene jest
 *       przeindeksowane rcznie.  Jest to znany bd i powinien by poprawiony w przyszych wersjach
 *       Hibernate Search. 
 */
public class IndexWhenActiveInterceptor implements EntityIndexingInterceptor<App> {
	
	/** 
	 * Ta metoda jest automatycznie uruchomiona gdy nowa encja typu App jest zapisywana przez Hibernate.  
	 * Gdy encja App jest aktywna, uyta jest domylna operacja Hibernate Search (tzn. dodanie do indeksu Lucene).
	 * Jeeli aplikacja jest nieaktywna, indeksowanie w Lucene si nie odbywa. 
	 */
	public IndexingOverride onAdd(App entity) {
		if(entity.isActive()) {
			return IndexingOverride.APPLY_DEFAULT;
		}
		return IndexingOverride.SKIP;
	}

	/**
	 * Ta metoda jest automatycznie uruchomiona gdy encja typu App jest usuwana przez Hibernate. Niezalenie
	 * od parametru "active" encji App, Hibernate Search ma wykona domyln akcj  (tj. usunicie wpisu z indeksu 
	 * Lucene, jeeli istnieje).
	 */
	public IndexingOverride onDelete(App entity) {
		return IndexingOverride.APPLY_DEFAULT;
	}

	/**
	 * Ta metoda jest automatycznie uruchomiona gdy encja typu App jest aktualizowana przez Hibernate.  Jeeli 
	 * aplikacja jest aktywna, indeks Lucene ma zosta zaktualizowany.  Jeeli aplikacja jest nieaktywna 
	 * encja ma by usunita z indeksu Lucene.
	 */
	public IndexingOverride onUpdate(App entity) {
		if(entity.isActive()) {
			return IndexingOverride.UPDATE;
		} else {
			return IndexingOverride.REMOVE;
		}
	}

	/**
	 * Ta metoda jest automatycznie uruchomiona gdy encja typu App znajduje si w kolekcji grupowo aktualizowanej
	 * przez Hibernate. Nie jest to istotne dla interceptora, wic zostaje uyta standardowa metoda onUpdate().
	 */
	public IndexingOverride onCollectionUpdate(App entity) {
		return onUpdate(entity);
	}

}
