package com.packtpub.hibernatesearch.domain;

import java.io.Serializable;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.ManyToMany;

import org.codehaus.jackson.annotate.JsonIgnore;
import org.hibernate.search.annotations.ContainedIn;
import org.hibernate.search.annotations.Field;

/**
 * Prosta klasa encji reprezentujca sprzt elektroniczny... posiada nazw, nazw producenta, oraz kolekcj aplikacji 
 * wspieranych przez urzdzenie.
 * 
 * Adnotacja @Entity informuje Hibernate by traktowao t klas jako samodzieln encj, z cyklem ycia nie uzalenionym
 * od innych klas encji. Niemniej jednak, w zwizku z tym, e ta klasa NIE JEST adnotowana @Indexed, Hibernate 
 * Search nie utworzy dla niej indeksu Lucene. Wszystkie informacje wskazane do indeksowania trafi do indeksu 
 * Lucene obiektu w ktrym si zawiera.
 */
@Entity
public class Device implements Serializable {

	private static final long serialVersionUID = 1L;
	
	/**
	 * Automatycznie generowany klucz gwny, nie ma potrzeby nadawania mu wartoci. 
	 */	
	@Id
  @GeneratedValue
  private Long id;
	
	/**
	 * Nazwa producenta urzdzenia.  Adnotacja @Column informuje Hibernate by zmapowa to pole do 
	 * kolumny w bazie danych, natomiast adnotacja @Field nazakuje Hibernate Search by zmapowa je jako pole 
	 * w indeksie Lucene.
	 */	
	@Column
	@Field
  private String manufacturer;
	
	/**
	 * Nazwa urzdzenia.  Adnotacja @Column informuje Hibernate by zmapowa to pole do 
	 * kolumny w bazie danych, natomiast adnotacja @Field nazakuje Hibernate Search by zmapowa je jako pole 
	 * w indeksie Lucene.
	 */	
	@Column
	@Field
  private String name;

	/**
	  * Kolekcja powizanych encji typu App, reprezentujcych aplikacje wspierane przez to urzdzenie.  
	 * 
	 * Adnotacja @ManyToMany informuje Hibernate, e aplikacja moe by wspierana przez wiele urzdze oraz, e urzdzenie 
	 * moe wspiera wiele aplikacji.  
	 * 
	 *      Pomimo, e relacja midzy aplikacjami i urzdzeniami jest dwustronna, zakadamy e klasa App bdzie nadrzdn
	 *      encj.  Element "mappedBy" informuje Hibernate ktra zmienna (np. App.supportedDevices) odwouje si do tej encji.
	 * 
	 *      Elementy "fetch" oraz "cascade" s ustawione identycznie jak dla App.supportedDevices.
	 *      By pozna wicej szczegw przejrzyj komentarze do klasy App.java.
	 * 
	 * Adnotacja Hibernate Search @ContainedIn jest odpowiednikiem @IndexedEmbedded (uytej dla App.SupportedDevices). 
	 * Instruuje Hibernate Search by doczyo informacje o urzdzeniach w indeksie Lucene encji App. 
	 * 
	 * Adnotacja @JsonIgnore jest uywana przez Jersey i Jackson w RESTowym serwisie pobierajcym informacje o aplikacji 
	 * do zapobiegania cyklicznym zalenociom podczas konwersji obiektu App do JSONowej reprezentacji. 
	 * "App.supportedDevices" byoby przekonwertowane do tablicy obiektw typu Device, kady z nich ma pole "supportedApps"
	 * ktre byoby odwzorowane na tablic aplikacji, ktre zawieraj tablic urzdze ...
	 * Za pomoc tej adnotacji sprawiamy, e "Device.supportedApps" nie jest konwertowane 
	 * dziki czemu zapobiegniemy cyklicznej zalenoci jednoczenie nie zaburzajc wywietlania wynikw.
	 */
	@ManyToMany(mappedBy="supportedDevices", fetch=FetchType.EAGER, cascade = { CascadeType.ALL })
	@ContainedIn
	@JsonIgnore
	private Set<App> supportedApps;

	/**
	 * Domylny, pusty konstruktor.
	 */	
	public Device() {
	}
	
	/**
	 * Wygodny konstruktor ustawiajcy wartoci wszystkich pl jednoczenie.  
	 */
	public Device(String manufacturer, String name, Set<App> supportedApps) {
		this.manufacturer = manufacturer;
		this.name = name;
		this.supportedApps = supportedApps;
	}
	
	//
	// GETTERY I SETTERY
	//
	
	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public String getManufacturer() {
		return manufacturer;
	}

	public void setManufacturer(String manufacturer) {
		this.manufacturer = manufacturer;
	}
	
	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public Set<App> getSupportedApps() {
		return supportedApps;
	}

	public void setSupportedApps(Set<App> supportedApps) {
		this.supportedApps = supportedApps;
	}

}
