package com.packtpub.hibernatesearch.servlet;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.hibernate.Session;
import org.hibernate.search.FullTextQuery;
import org.hibernate.search.FullTextSession;
import org.hibernate.search.Search;
import org.hibernate.search.query.dsl.QueryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.packtpub.hibernatesearch.domain.App;
import com.packtpub.hibernatesearch.util.StartupDataLoader;

/**
 * Servlet implementujcy kontroler/model obsugujcy przeszukiwanie i renderowanie wynikw przy pomocy widoku w JSP/JSTL.
 * Adnotacja @Webservlet, dostpna w specyfikacji servletw w wersji 3.0, mapuje ten servlet do URLa "search (np. 
 * "http://localhost:8080/search"). We wczeniejszych wersjach specyfikacji ta konfiguracja znajdowaaby si w pliku "web.xml".
 * Podstawowa logika tej operacji wyszukiwania moe by zrefaktoryzowana w celu adaptacji do aplikacji napisanych w Springu, JSFie
 * lub dowolnym innym Javowym frameworku do aplikacji internetowych.
 */
@SuppressWarnings("serial")
@WebServlet("search")
public class SearchServlet extends HttpServlet {

	/**
	 * W tej metodzie zaimplementowane zostay gwne funkcjonalnoci wyszukiwania dla tego servletu. Metoda jest automatycznie wywoywana
	 * przy kadym wywoaniu dania HTTP POST do zmapowanego adresu URL.
	 */
	@SuppressWarnings("unchecked")
	@Override	
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		
		Logger logger = LoggerFactory.getLogger(SearchServlet.class);
		
		// Pobierz sowa kluczowe uytkownika.  Pobierz opcjonalne parametry stronicowania i sortowania lub uyj domylnych wartoci jeli nie podano.
		String searchString = request.getParameter("searchString") != null ? request.getParameter("searchString").trim() : "";
		String sortField = request.getParameter("sortField") != null ? request.getParameter("sortField").trim() : "relevance";
		int firstResult = request.getParameter("firstResult") != null ? Integer.parseInt(request.getParameter("firstResult")) : 0;
		logger.info("Wyszukuj po [" + searchString + "], sortuj [" + sortField + "], wyniki rozpoczynam od [" +  firstResult + "]");

		// Rozpocznij sesj Hibernate.
		Session session = StartupDataLoader.openSession();
		
		// Utwrz wrapper Hibernate Search wok czystej sesji Hibernate
		FullTextSession fullTextSession = Search.getFullTextSession(session);

		// Rozpocznij transkacj. W tym przypadku nie jest to niezbdne, ale jest tzw. dobr praktyk.
		fullTextSession.beginTransaction();

		// Utwrz obiekt typu QueryBuilder pochodzcy z Hibernate Search i utwrz go dla odpowiedniego indeksu Lucene (w naszym przypadku indeksu "App").
		QueryBuilder queryBuilder = fullTextSession.getSearchFactory().buildQueryBuilder().forEntity( App.class ).get();
		
		// Uyj QueryBuildera by zbudowa w Lucene zapytanie z uyciem sw kluczowych, porwnujce sowa kluczowe wprowadzone przez uytkownika z polami 
		// name i description w klasie App, pola name powizanych aplikacji oraz pola comments wbudowanych obiektw klasy CustomerReview.
		org.apache.lucene.search.Query luceneQuery = null;
		if(searchString.length() > 2 && searchString.startsWith("\"") && searchString.endsWith("\"")) {
			
			// Jeeli sowa kluczowe wprowadzone przez uytkownika s umieszczone w cudzysowach, uyj wyszukiwania z uyciem frazy
			String unquotedSearchString = searchString.substring(1, searchString.length() - 1);
			luceneQuery = queryBuilder
					.phrase()
					.onField("name").andField("description").andField("supportedDevices.name").andField("customerReviews.comments")
					.sentence(unquotedSearchString)
					.createQuery();			
		} else {
			
			// Jeeli sowa kluczowe wprowadzone przez uytkownika nie s umieszczone w cudzysowach, uyj rozmytego wyszukiwania z uyciem sw kluczowych
 			luceneQuery = queryBuilder
					.keyword()
					.fuzzy()
					.withThreshold(0.7f)
					.onFields("name", "description", "supportedDevices.name", "customerReviews.comments")
					.matching(searchString)
					.createQuery();
		}
		FullTextQuery hibernateQuery = fullTextSession.createFullTextQuery(luceneQuery, App.class);  // moe by rzutowane na "org.hibernate.Query"

		// Dodaj opcjonalne kryteria sortowania, jeeli nie zostay ustawione, sortuj wg wanoci
		if(sortField.equals("name")) {
			Sort sort = new Sort(new SortField("sorting_name", SortField.STRING));
			hibernateQuery.setSort(sort);
		} else if(sortField.equals("name-reverse")) {
			Sort sort = new Sort(new SortField("sorting_name", SortField.STRING, true));
			hibernateQuery.setSort(sort);
		}

		// Pobierz szacowan liczb wynikw (UWAGA: nie jest 100% dokadna, ale nie wymaga odpytania bazy danych 
		// co moe by kosztowne przy duych zbiorach danych).
		int resultSize = hibernateQuery.getResultSize();

		// Wykonaj wyszukiwanie... ograniczone do 5 wynikw, oraz punktem pocztkowym w miejscu gdzie poprzednia strona zakoczya
		// prezentacj wynikw (domylnie od pocztku)
		logger.info("Query string == " + hibernateQuery.getQueryString());
		hibernateQuery.setFirstResult(firstResult);
		hibernateQuery.setMaxResults(5);
		List<App> apps = hibernateQuery.list();
		
		// Odcz wyniki z sesji Hibernate (by unikn niechcianej interakcji midzy warstw prezentacji i Hibernate,
		// gdy powizane urzdzenia lub wbudowane obiekty typu CustomerReview s przegldane).
		fullTextSession.clear();

		// Umie wyniki wyszukiwania w obiekcie dania HTTP, wraz z parametrami sortowania i stronicowania
		request.setAttribute("searchString", searchString);
		request.setAttribute("sortField", sortField);
		request.setAttribute("apps", apps);
		request.setAttribute("resultSize", resultSize);
		request.setAttribute("firstResult", firstResult);

		// Oprnij i zamknij sesj Hibernate.
		fullTextSession.getTransaction().commit();
		session.close();
		
		// Przeka danie HTTP, wraz z wynikami wyszukiwania, do widoku zbudowanego JSP/JSTL w celu wyrenderowania strony z wynikami
		getServletContext().getRequestDispatcher("/WEB-INF/pages/search.jsp").forward(request, response);
	}

	/**
	 * Ta metoda jest wywoywana za kadym razem gdy zostanie wywoane danie HTTP GET do zmapowanego URLa.  Dla naszego servletu nie ma znaczenia, 
	 * czy zostanie wywoany metod POST czy GET. W zwizku z tym przekierowujemy danie do metody "doPost()".
	 */
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {	
		this.doPost(request, response);
	}
	
}
