package com.packtpub.hibernatesearch.domain;

import java.io.Serializable;
import java.util.Set;

/**
 * Prosta klasa encji reprezentujca aplikacj komputerow ... z nazw, dugim opisem 
 * i nazw pliku uytego jako ikona. Klasa posiada flag "active" okrelajc, czy dana aplikacja
 * ma by wyszukiwalna.  
 * 
 * Ten wariant klasy App jest mapowany do bazy danych za pomoc klasycznego podejcia opartego na plikach XML.
 * W Mavenowej strukturze projektu wszystkie pliki XML s umieszczone w "src/main/resources".  Hibernate 
 * automatycznie przeglda plik "hibernate.cfg.xml", ktry odwouje si do "com/packtpub/hibernatesearch/domain/App.hbm.xml"
 * by zaadowa informacje o mappowaniu encji.  Rezultaty s identyczne jak w przypadku uycia adnotacji @Entity.
 * 
 * Pomimo, e adnotacja @Entity nie jest uywana, konfiguracja Hibernate Search jest nadal oparta o adnotacje.
 * Adnotacja @Indexed informuje Hibernate Search by mapowaa encj na indeks Lucene.  Element "interceptor" 
 * ustawia warunki indeksowania tak, by encje z flag "active" rwn "false" nie byy indeksowane  
 * (po wicej szczegw zajrzyj do klasy "com.packtpub.hibernatesearch.util.IndexWhenActiveInterceptor").
 */
public class App implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * Klucz gwny, skonfigurowany w pliku "App.hbm.xml" tak, by by generowany automatycznie podczas tworzenia nowej instancji
	 * 
	 * Adnotacja @DocumentId instruuje Hibernate Search by uyo tej wartoci jako klucz gwny podczas indeksowania encji w Lucene.  
	 * Jeeli uywasz adnotacji do mapowania na tabele w bazie danych, Hibernate Search skorzysta z adnotacji @Id. Uywajc mapowania XML
	 * musisz zadeklarowa @DocumentId.
	 */
    private Long id;

	/**
	 * Nazwa aplikacji zrozumiaa dla ludzi.  Zamiast adnotacji @Column uywamy informacji z pliku "App.hbm.xml", ale 
	 * wci uywamy adnotacji @Field by poinformowa Hibernate Search, e pole ma by oznaczone jako wyszukiwalne w indeksie Lucene.
	 */
    private String name;	
	
	/**
	 * Duszy, bardziej szczegowy opis aplikacji.  Zamiast adnotacji @Column uywamy informacji z pliku "App.hbm.xml", ale 
	 * wci uywamy adnotacji @Field by poinformowa Hibernate Search, e pole ma by oznaczone jako wyszukiwalne w indeksie Lucene.
	 */
    private String description;

	/**
	 * Nazwa pliku obrazka doczonego do opisu aplikacji. Plik powinien si znajdowa w podkatalogu
	 * "images/apps/<obrazek>". Obrazek nie jest adnotowany @Field poniewa nie bdziemy wyszukiwa aplikacji
	 * na podstawie obrazkw.
	 */	
    private String image;

	/**
	 * Flaga okrelajca, czy aplikacja ma by wyszukiwalna.
	 */
    private boolean active;

	/**
	 * Kolekcja powizanych encji typu Device, reprezentujcych urzdzenia na ktrych mona uruchomi aplikacj.
	 * 
	 * Mapowanie Hibernate odbywa si w pliku "App.hbm.xml", uywamy elementu XML "<many-to-many>"
	 * zamiast adnotacji @ManyToMany.  Niemniej jednak pobieranie oraz kaskadowanie opisane w komentarzach do kodu w katalogu 
	 * "rozdzial2" maj rwnie zastosowanie w tym przypadku.
	 * 
	 * Adnotacja @IndexedEmbedded informuje Hibernate Search by indeksowa powizane encje.  
	 * Klasa Device moe (ale nie musi) posiada indeks Lucene (w zalenoci od tego czy klasa jest adnotowana @Indexed), 
	 * ale i tak instancje Device powizane z aplikacj bd przechowywane w indeksie Lucene klasy App.  
	 */
    //@IndexedEmbedded(depth=1)
    private Set<Device> supportedDevices;

	/**
	 * Kolekcja wbudowanych obiektw prezentujcych opinie uytkownikw o aplikacji. Wbudowane obiekty rni si od powizanych encji 
	 * tym, e z chwil skasowania obiektu, z ktrym s powizane one rwnie zostaj usunite. 
	 * 
	 * Mapowanie Hibernate odbywa si w pliku "App.hbm.xml", uywamy elementu XML "<composite-element>"
	 * zamiast adnotacji @ElementCollection.  Niemniej jednak pobieranie oraz kaskadowanie opisane w komentarzach do kodu w katalogu 
	 * "rozdzial2" maj rwnie zastosowanie w tym przypadku.
	 * 
	 * Adnotacja @IndexEmbedded ma zdefiniowany opcjonalny element "includePaths" by wskaza, ktre pola wbudowanych obiektw 
	 * bd indeksowane w encji nadrzdnej.  Innymi sowy pole "comments" bdzie jedyn wartoci klasy CustomerReview doczon
	 * do indeksu Lucene klasy App.  
	 */
    //@IndexedEmbedded(depth=1, includePaths = { "comments" })
	private Set<CustomerReview> customerReviews;

	/**
	 * Domylny pusty konstruktor.
	 */
	public App() {}

	/**
	 * Wygodny konstruktor, ustawiajcy zawarto wszystkich pl w jednym kroku.  
	 * 
	 * Parametr "active" jest domylnie ustawiony na warto "true".  Aby dezaktywowa aplikacj,
	 * uyj settera dla tego pola. 
	 */
	public App(String name, String image, String description) {
		this.name = name;
		this.image = image;
		this.description = description;
		this.active = true;
	}

	//
	// GETTERY I SETTERY
	//
	
	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getImage() {
		return image;
	}

	public void setImage(String image) {
		this.image = image;
	}

	public Set<Device> getSupportedDevices() {
		return supportedDevices;
	}

	public void setSupportedDevices(Set<Device> supportedDevices) {
		this.supportedDevices = supportedDevices;
	}

	public boolean isActive() {
		return active;
	}

	public void setActive(boolean active) {
		this.active = active;
	}

	public Set<CustomerReview> getCustomerReviews() {
		return customerReviews;
	}

	public void setCustomerReviews(Set<CustomerReview> customerReviews) {
		this.customerReviews = customerReviews;
	}

}
