package com.pragprog.aebook.persistchat.client;

import java.util.Date;
import java.util.List;

import com.google.gwt.core.client.EntryPoint;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.pragprog.aebook.persistchat.ChatMessage;

/**
 * Entry point classes define <code>onModuleLoad()</code>.
 */
public class Chat implements EntryPoint {
	/**
	 * The message displayed to the user when the server cannot be reached or
	 * returns an error.
	 */
	private static final String SERVER_ERROR = "An error occurred while "
			+ "attempting to contact the server. Please check your network "
			+ "connection and try again.";

	/**
	 * Create a remote service proxy to talk to the server-side Greeting service.
	 */
	private final ChatServiceAsync chatService = GWT
			.create(ChatService.class);
	
	private TextArea text;

	/**
	 * This is the entry point method.
	 */
	public void onModuleLoad() {
//START: MainPanel	    
        final VerticalPanel mainVert = new VerticalPanel(); // <callout id="co.mainPanel"/>

        final VerticalPanel topPanel = new VerticalPanel(); // <callout id="co.subpanels"/>
        final HorizontalPanel midPanel = new HorizontalPanel();
        final HorizontalPanel bottomPanel = new HorizontalPanel();
        mainVert.add(topPanel); // <callout id="co.populate_main"/>
        mainVert.add(midPanel);
        mainVert.add(bottomPanel);
//END: MainPanel
        
//START: TopPanel
        final Label title = new Label("AppEngine Chat"); // <callout id="co.top.labels"/>
        final Label subtitle = new Label(new Date().toString());
        title.addStyleName("title"); // <callout id="co.top.style"/>
        topPanel.add(title);// <callout id="co.top.pop"/>
        topPanel.add(subtitle);
//END: TopPanel        

//START: MidPanel
        final VerticalPanel chatList = new VerticalPanel(); // <callout id="co.chatlist-panel"/>
        chatList.setBorderWidth(2); 
        final Label chatLabel = new Label("Chats");
        chatLabel.addStyleName("emphasized");//<callout id="co.chatlist-style"/>
        chatList.add(chatLabel);
        chatList.setWidth("10em");//<callout id="co.chatlist-width"/>
        populateChats(chatList); //<callout id="co.chatlist-populate"/>
        // "TextArea text" is defined as a field of the class, so that
        // the textarea can be referenced by handler methods.
        text = new TextArea();//<callout id="co.chat-text-area"/>
        text.addStyleName("messages");
        text.setWidth("60em");
        text.setHeight("20em");
        midPanel.add(chatList);
        midPanel.add(text);
//END: MidPanel

//START: BottomPanel        
        final Label label = new Label("Enter Message:");
        label.addStyleName("bold");
        final TextBox messageBox = new TextBox(); 
        messageBox.setWidth("60em");
        final Button sendButton = new Button("send");
        bottomPanel.add(label);
        bottomPanel.add(messageBox);
        bottomPanel.add(sendButton);
        setupSendMessageHandlers(sendButton, messageBox);
//END: BottomPanel        

//START: GoLive
        RootPanel.get().add(mainVert); 

        // focus the cursor on the message box.
        messageBox.setFocus(true); 
        messageBox.selectAll();
        setupTimedUpdate();
	}
//END: GoLive

	//START: Timed
	private void setupTimedUpdate() {
        // Create a new timer
        Timer elapsedTimer = new Timer() {
            public void run() {                
                chatService.getMessagesSince(getCurrentChat(), lastMessageTime,
                        new MessageListCallback());
            }
        };
        // Schedule the timer for every 1/2 second (500 milliseconds)
        elapsedTimer.scheduleRepeating(500);
	}
//END: Timed	

//START: Populate
    /**
     * Sets up a call and callback to retrieve the list of available chats. When
     * the server responds, this will create the link widgets and add them to
     * the chatListPanel.
     * 
     * @param chatListPanel
     */
    public void populateChats(final VerticalPanel chatListPanel) {
        chatService.getChats(new AsyncCallback<List<String>>() { //<callout id="co.sendgetchats"/>
            public void onFailure(Throwable caught) { //<callout id="co.getchats.error"/>
                chatListPanel.add(new Label("Couldn't retrieve chats: " + caught));
            }

            public void onSuccess(List<String> chats) {//<callout id="co.getchats.success"/>
                for (String chat : chats) {
                    Button chatButton = new Button(chat);
                    chatListPanel.add(chatButton);
                    Chat.this.setupChatClickHandler(chatButton,//<callout id="co.getchats.sethandler"/>
                            chat);
                }
                setCurrentChat(chats.get(0));

            }
        });
    }
//END: Populate    

 //START: ChatClickHandler
    protected void setupChatClickHandler(final Button chatButton, final String chat) {
        chatButton.addClickHandler(new ClickHandler() { 
            public void onClick(ClickEvent event) {
                setCurrentChat(chat);
                text.setText("Current chat: " + chat + "\n");
                currentChat = chat;
                chatService.getMessages(currentChat, new MessageListCallback());
            }
        });
    }
//END: ChatClickHandler

//START: MessageListCallback    
    public class MessageListCallback implements AsyncCallback<List<ChatMessage>> {

        public void onFailure(Throwable caught) {
        }

        public void onSuccess(List<ChatMessage> result) {
            addNewMessages(result);
        }        
    }
    
    protected void addNewMessages(List<ChatMessage> newMessages) {
        StringBuilder content = new StringBuilder();
        content.append(text.getText());
        for (ChatMessage cm : newMessages) {
            content.append(renderChatMessage(cm));
        }
        text.setText(content.toString());
    }
    
    protected String renderChatMessage(ChatMessage msg) {
        Date d = new Date(msg.getDate());
        String dateStr = d.getMonth() + "/" + d.getDate() + " " +
            d.getHours() + ":" + d.getMinutes() + "." +
            d.getSeconds();
        return "[From: " + msg.getSenderName() + " at " +
            dateStr  + "]: " + msg.getMessage() + "\n";  
    }

    protected void addNewMessage(ChatMessage newMessage) {
        text.setText(text.getText() + renderChatMessage(newMessage));
    }
//END: MessageListCallback        



//START: handlers    
    private void setupSendMessageHandlers(final Button sendButton,
            final TextBox messageBox) {
        // Create a handler for the sendButton and nameField
        class SendMessageHandler implements ClickHandler,// <callout id="co.send-handler"/>
                KeyUpHandler {
            /** Fired when the user clicks on the sendButton. */
            public void onClick(ClickEvent event) {// <callout id="co.handle-click"/>
                sendMessageToServer();
            }

            /** Fired when the user types in the nameField. */
            public void onKeyUp(KeyUpEvent event) {// <callout id="co.onkeyup"/>
                if (event.getNativeKeyCode() == KeyCodes.KEY_ENTER) {
                    sendMessageToServer();
                }
            }

            /** Send a chat message to the server. */
            private void sendMessageToServer() { // <callout id="co.send-to-server"/>
                ChatMessage chatmsg = new ChatMessage(user,
                        messageBox.getText(), getCurrentChat());
                messageBox.setText("");
                chatService.postMessage(chatmsg,
                        new AsyncCallback<Void>() {
                    public void onFailure(Throwable caught) {
                        Chat.this.addNewMessage(new ChatMessage(
                                "System", "Error sending message: " +
                                caught.getMessage(),
                                getCurrentChat()));
                    }

                    public void onSuccess(Void v) {
                        chatService.getMessagesSince(getCurrentChat(), 
                                lastMessageTime,
                                new MessageListCallback());
                    }
                });
            }
        }
        SendMessageHandler handler = new SendMessageHandler();// <callout id="co.register-callback"/>
        sendButton.addClickHandler(handler);
        messageBox.addKeyUpHandler(handler);
    }
//END: handlers        

	protected void AddPostedMessages(List<ChatMessage> newMessages) {
		// TODO Auto-generated method stub
		
	}
}
