"""Obiektowy framework TensorGraph."""

import numpy as np
import tensorflow as tf
import copy
import multiprocessing
import os
import re
import threading
from collections import Sequence

import pickle
import threading
import time

import numpy as np
import os
import six
import tensorflow as tf
import tempfile

class TensorGraph(object):

  def __init__(self,
               batch_size=100,
               random_seed=None,
               graph=None,
               learning_rate=0.001,
               model_dir=None,
               **kwargs):
    """
    Parametry
    ----------
    batch_size: liczba całkowita
      domyślna wielkość grupy dla potrzeb szkolenia i ewaluacji
    graph: tensorflow.Graph
      graf, w którym tworzone będą obiekty Tensorflow.  Przy None tworzony jest
      nowy wykres.
    learning_rate: liczba zmiennoprzecinkowa lub LearningRateSchedule
      współczynnik uczenia do wykorzystania w celu optymalizacji
    kwargs
    """

    # Zarządzanie warstwami
    self.layers = dict()
    self.features = list()
    self.labels = list()
    self.outputs = list()
    self.task_weights = list()
    self.loss = None
    self.built = False
    self.optimizer = None
    self.learning_rate = learning_rate

    # Pojedyncze miejsce na obiekty Tensor, które się nie serializują
    # Więcej szczegółów na temat konstrukcji "leniwych": patrz TensorGraph._get_tf()
    self.tensor_objects = {
        "Graf": graph,
        #"train_op": None,
    }
    self.global_step = 0

    self.batch_size = batch_size
    self.random_seed = random_seed
    if model_dir is not None:
      if not os.path.exists(model_dir):
        os.makedirs(model_dir)
    else:
      model_dir = tempfile.mkdtemp()
      self.model_dir_is_temp = True
    self.model_dir = model_dir
    self.save_file = "%s/%s" % (self.model_dir, "model")
    self.model_class = None

  def _add_layer(self, layer):
    if layer.name is None:
      layer.name = "%s_%s" % (layer.__class__.__name__, len(self.layers) + 1)
    if layer.name in self.layers:
      return
    if isinstance(layer, Input):
      self.features.append(layer)
    self.layers[layer.name] = layer
    for in_layer in layer.in_layers:
      self._add_layer(in_layer)

  def topsort(self):

    def add_layers_to_list(layer, sorted_layers):
      if layer in sorted_layers:
        return
      for in_layer in layer.in_layers:
        add_layers_to_list(in_layer, sorted_layers)
      sorted_layers.append(layer)

    sorted_layers = []
    for l in self.features + self.labels + self.task_weights + self.outputs:
      add_layers_to_list(l, sorted_layers)
    add_layers_to_list(self.loss, sorted_layers)
    return sorted_layers

  def build(self):
    if self.built:
      return
    with self._get_tf("Graf").as_default():
      self._training_placeholder = tf.placeholder(dtype=tf.float32, shape=())
      if self.random_seed is not None:
        tf.set_random_seed(self.random_seed)
      for layer in self.topsort():
        with tf.name_scope(layer.name):
          layer.create_tensor(training=self._training_placeholder)
      self.session = tf.Session()

      self.built = True

  def set_loss(self, layer):
    self._add_layer(layer)
    self.loss = layer

  def add_output(self, layer):
    self._add_layer(layer)
    self.outputs.append(layer)

  def set_optimizer(self, optimizer):
    """Określenie optymalizatora używanego przy dopasowywaniu."""
    self.optimizer = optimizer

  def get_layer_variables(self, layer):
    """Pobieranie listy możliwych do wyszkolenia zmiennych w warstwie grafu."""
    if not self.built:
      self.build()
    with self._get_tf("Graf").as_default():
      if layer.variable_scope == "":
        return []
      return tf.get_collection(
          tf.GraphKeys.TRAINABLE_VARIABLES, scope=layer.variable_scope)

  def get_global_step(self):
    return self._get_tf("KrokGlobalny")

  def _get_tf(self, obj):
    """Pobieranie podstawowych elementów składowych TensorFlow.

    Parametry
    ----------
    obj: str
      Jeśli "Graf", zwraca instancję tf.Graph. Jeśli "Optymalizator", zwraca
      optymalizator. Jeśli "train_op", zwraca operację treningu. Jeśli
      "KrokGlobalny", zwraca krok globalny.  
    Zwracane wartości
    -------
    Obiekt TensorFlow

    """

    if obj in self.tensor_objects and self.tensor_objects[obj] is not None:
      return self.tensor_objects[obj]
    if obj == "Graf":
      self.tensor_objects["Graf"] = tf.Graph()
    elif obj == "Optymalizator":
      self.tensor_objects["Optymalizator"] = tf.train.AdamOptimizer(
          learning_rate=self.learning_rate,
          beta1=0.9,
          beta2=0.999,
          epsilon=1e-7)
    elif obj == "KrokGlobalny":
      with self._get_tf("Graf").as_default():
        self.tensor_objects["KrokGlobalny"] = tf.Variable(0, trainable=False)
    return self._get_tf(obj)

  def restore(self):
    """Przeładowanie wartości wszystkich zmiennych z ostatniego pliku punktu
	   kontrolnego."""
    if not self.built:
      self.build()
    last_checkpoint = tf.train.latest_checkpoint(self.model_dir)
    if last_checkpoint is None:
      raise ValueError("Nie znaleziono punktu kontrolnego")
    with self._get_tf("Graf").as_default():
      saver = tf.train.Saver()
      saver.restore(self.session, last_checkpoint)

  def __del__(self):
    pass

class Layer(object):

  def __init__(self, in_layers=None, **kwargs):
    if "name" in kwargs:
      self.name = kwargs["name"]
    else:
      self.name = None
    if in_layers is None:
      in_layers = list()
    if not isinstance(in_layers, Sequence):
      in_layers = [in_layers]
    self.in_layers = in_layers
    self.variable_scope = ""
    self.tb_input = None

  def create_tensor(self, in_layers=None, **kwargs):
    raise NotImplementedError("Podklasy muszą implementować dla siebie")

  def _get_input_tensors(self, in_layers):
    """Pobranie tensorów wejściowych do swojej warstwy.

    Parametry
    ----------
    in_layers: lista warstw lub tensorów
      wejścia przesłane do create_tensor(). Przy None, zamiast tego zostaną użyte
      wejścia tej warstwy.
    """
    if in_layers is None:
      in_layers = self.in_layers
    if not isinstance(in_layers, Sequence):
      in_layers = [in_layers]
    tensors = []
    for input in in_layers:
      tensors.append(tf.convert_to_tensor(input))
    return tensors

def _convert_layer_to_tensor(value, dtype=None, name=None, as_ref=False):
  return tf.convert_to_tensor(value.out_tensor, dtype=dtype, name=name)


tf.register_tensor_conversion_function(Layer, _convert_layer_to_tensor)

class Dense(Layer):

  def __init__(
      self,
      out_channels,
      activation_fn=None,
      biases_initializer=tf.zeros_initializer,
      weights_initializer=tf.contrib.layers.variance_scaling_initializer,
      **kwargs):
    """Tworzenie gęstej warstwy.

    Inicjalizatory wag i obciążeń są określane przez wywoływalne obiekty, które konstruują
    i zwracają inicjalizator Tensorflow, gdy są wywoływane bez argumentów.  Zazwyczaj będzie
    to albo sama klasa inicjalizatora (jeśli konstruktor nie wymaga argumentów), albo TFWrapper
    (w przeciwnym przypadku).

    Parametry
    ---------
    out_channels: liczba całkowita
      liczba wartości wyjściowych
    activation_fn: obiekt
      funkcja aktywacji Tensorflow, która ma być zastosowana do wyjścia
    biases_initializer: wywoływalny obiekt
      inicjalizator wartości obiążeń.  Może to być None, w tym przypadku warstwa
      nie będzie zawierać obciążeń.
    weights_initializer: wywoływalny obiekt
      inicjalizator dla wartości wagowych
    """
    super(Dense, self).__init__(**kwargs)
    self.out_channels = out_channels
    self.out_tensor = None
    self.activation_fn = activation_fn
    self.biases_initializer = biases_initializer
    self.weights_initializer = weights_initializer

  def create_tensor(self, in_layers=None, **kwargs):
    inputs = self._get_input_tensors(in_layers)
    if len(inputs) != 1:
      raise ValueError("Gęsta warstwa może mieć tylko jedno wejście")
    parent = inputs[0]
    if self.biases_initializer is None:
      biases_initializer = None
    else:
      biases_initializer = self.biases_initializer()
    out_tensor = tf.contrib.layers.fully_connected(parent,
                                                   num_outputs=self.out_channels,
                                                   activation_fn=self.activation_fn,
                                                   biases_initializer=biases_initializer,
                                                   weights_initializer=self.weights_initializer(),
                                                   reuse=False,
                                                   trainable=True)
    self.out_tensor = out_tensor
    return out_tensor

class Squeeze(Layer):

  def __init__(self, in_layers=None, squeeze_dims=None, **kwargs):
    self.squeeze_dims = squeeze_dims
    super(Squeeze, self).__init__(in_layers, **kwargs)

  def create_tensor(self, in_layers=None, **kwargs):
    inputs = self._get_input_tensors(in_layers)
    parent_tensor = inputs[0]
    out_tensor = tf.squeeze(parent_tensor, squeeze_dims=self.squeeze_dims)
    self.out_tensor = out_tensor
    return out_tensor

class BatchNorm(Layer):

  def __init__(self, in_layers=None, **kwargs):
    super(BatchNorm, self).__init__(in_layers, **kwargs)

  def create_tensor(self, in_layers=None, **kwargs):
    inputs = self._get_input_tensors(in_layers)
    parent_tensor = inputs[0]
    out_tensor = tf.layers.batch_normalization(parent_tensor)
    self.out_tensor = out_tensor
    return out_tensor

class Flatten(Layer):
  """Spłaszczanie każdego wymiaru z wyjątkiem pierwszego"""

  def __init__(self, in_layers=None, **kwargs):
    super(Flatten, self).__init__(in_layers, **kwargs)

  def create_tensor(self, in_layers=None, **kwargs):
    inputs = self._get_input_tensors(in_layers)
    if len(inputs) != 1:
      raise ValueError("Tylko jeden element nadrzędny do spłaszczenia")
    parent = inputs[0]
    parent_shape = parent.get_shape()
    vector_size = 1
    for i in range(1, len(parent_shape)):
      vector_size *= parent_shape[i].value
    parent_tensor = parent
    out_tensor = tf.reshape(parent_tensor, shape=(-1, vector_size))
    self.out_tensor = out_tensor
    return out_tensor

class SoftMax(Layer):

  def __init__(self, in_layers=None, **kwargs):
    super(SoftMax, self).__init__(in_layers, **kwargs)

  def create_tensor(self, in_layers=None, **kwargs):
    inputs = self._get_input_tensors(in_layers)
    if len(inputs) != 1:
      raise ValueError("Wymagany pojedynczy element nadrzędny Softmax")
    parent = inputs[0]
    out_tensor = tf.contrib.layers.softmax(parent)
    self.out_tensor = out_tensor
    return out_tensor

class Input(Layer):

  def __init__(self, shape, dtype=tf.float32, **kwargs):
    self._shape = tuple(shape)
    self.dtype = dtype
    super(Input, self).__init__(**kwargs)

  def create_tensor(self, in_layers=None, **kwargs):
    if in_layers is None:
      in_layers = self.in_layers
    out_tensor = tf.placeholder(dtype=self.dtype, shape=self._shape)
    self.out_tensor = out_tensor
    return out_tensor
