"""Python Sandbox Developer."""
import io
import logging
import os
import uuid
from contextlib import contextmanager, redirect_stdout
from logging import FileHandler
from pathlib import Path
from typing import Literal

import pip
from langchain import LLMChain, PromptTemplate
from langchain.chains.base import Chain
from langchain.llms import FakeListLLM
from langchain.tools.python.tool import sanitize_input
from pip._internal.exceptions import InstallationError
from pydantic import BaseModel, Field

logging.basicConfig(encoding="utf-8", level=logging.INFO)
DEV_PROMPT = (
    "Jesteś inżynierem oprogramowania, który pisze kod w Pythonie dla danego zadania lub osiągający określone cele. "
    "Napisz kod w Pythonie dla tego zadania: {task}"
    "Używaj składni PEP8 i dodawaj komentarze!"
)


class PythonExecutorInput(BaseModel):
    code: str = Field()


def meaningful_output(func):
    def wrapper(*args, **kwargs):
        func_output = func(*args, **kwargs)
        print(func_output)
        if len(func_output.strip()) > 0:
            return f"Kod zwrócił taki wynik:\n" \
                   f"{func_output}"
        else:
            return "Kod nic nie zwrócił."
    return wrapper


@contextmanager
def set_directory(path: Path):
    """Ustawia folder roboczy w obrębie kontekstu

    Parametry:
        * path (Path): ścieżka do folderu
    """
    origin = Path().absolute()
    try:
        os.chdir(path)
        yield
    finally:
        os.chdir(origin)


class PythonDeveloper:
    """Execution environment for Python code."""

    def __init__(
            self,
            llm_chain: Chain,
            path: str = "dev",
            audit_file: str = "audit.log",
            do_sanitize_input: bool = True,
            save_intermediate_steps: bool = False
    ):
        self.save_intermediate_steps = save_intermediate_steps
        self.llm_chain = llm_chain
        self.path = path
        self.create_directory()
        self.logger = logging.getLogger()
        self.logger.addHandler(
            self.setup_audit_trail(audit_file=audit_file)
        )
        self.do_sanitize_input = do_sanitize_input

    def write_code(self, task: str) -> str:
        """Na podstawie opisu zadania napisz kod w Pythonie.

        Jeśli wymagane są pośrednie kroki, kod zapusz
        w osobnym pliku Pythona.
        """
        code = self.llm_chain.run(task)
        if self.save_intermediate_steps:
            self.write_file("", code, "w")
        return code

    @staticmethod
    def setup_audit_trail(audit_file: str) -> FileHandler:
        """Ustaw rejestr zapisujący wszystkie wywołania."""
        formatter = logging.Formatter(
            "%(asctime)s | %(levelname)s | %(message)s"
        )
        file_handler = logging.FileHandler(audit_file)
        file_handler.setLevel(logging.DEBUG)
        file_handler.setFormatter(formatter)
        return file_handler

    def install_package(
            self,
            module_not_found_error: ModuleNotFoundError
    ) -> bool:
        """Zainstaluj paczkę.

        Zwraca wartość True, jeśli instalacja zakończyła się powodzeniem.
        Uwaga: Tutaj możemy zaiplementować logikę dla wirtualnych środowisk.
        """
        try:
            package = str(module_not_found_error).strip().split(" ")[-1].strip("'")
            self.logger.info(f"Instalowanie {package}")
            pip.main(['install', package])
            return True
        except InstallationError as ex:
            # Każdy inny błąd kończy działanie programu tutaj.
            self.logger.exception(ex)
            return False

    @meaningful_output
    def run(
            self,
            task: str,
            filename: str = "main.py",
            mode: Literal["w", "a"] = "w"
    ) -> str:
        """Wyegenruj i wykonaj kod Pythona.

        Zwraca wynik wykonanego kodu.
        """
        self.logger.info(f"Zadanie:\n{task}")
        code = self.write_code(task)
        self.logger.info(f"Kod:\n{code}")
        if self.do_sanitize_input:
            code = sanitize_input(code)

        self.write_file(code=code, filename=filename, mode=mode)
        # import executor; można zaimportować również bibliotekę pylint
        try:
            # with DirectorySandbox(self.path):
            return self.execute_code(code, filename)
        except (ModuleNotFoundError, NameError) as ex:
            return str(ex)
        except SyntaxError as ex:
            return f"This is not valid Python code! Exception thrown: {ex}"
        except FileNotFoundError as ex:
            # Jeśli to obraz, można dodać narzędzie tworzące obrazy.
            return f"Ten plik nie istnieje!\n{ex}"
        except SystemExit as ex:
            self.logger.warning(ex)
            return str(ex)

    def execute_code(self, code: str, filename: str) -> str:
        """Execute a python code."""
        try:
            with set_directory(Path(self.path)):
                ns = dict(__file__=filename, __name__="__main__")
                function = compile(code, "<>", "exec")
                with redirect_stdout(io.StringIO()) as f:
                    exec(function, ns)
                    return f.getvalue()
        except ModuleNotFoundError as ex:
            if self.install_package(ex):
                return self.execute_code(code, filename)
            raise ex

    def write_file(
            self,
            filename: str,
            code: str,
            mode: Literal["w", "a"] = "w"
    ) -> Path:
        """Zapisz kod na dysku.

        Jeśli plik jest pustym łańcuchem znaków, zapisz go
        w pliku o losowej nazwie. Może się przydać jako
        krok pośredni.

        Returns the path to the new file.
        """
        if not filename:
            filename = str(uuid.uuid4()) + ".py"
        fullpath = Path(self.path) / filename
        with open(fullpath, mode, encoding="utf-8") as f:
            f.write(code)

        return fullpath

    def create_directory(self):
        """Utwórz folder projektu."""
        os.makedirs(self.path, exist_ok=True)


if __name__ == "__main__":
    software_prompt = PromptTemplate.from_template(DEV_PROMPT)
    # Ostrożnie: jeśli masz błędną specyfikację modelu, możesz nie uzyskać żadnego kodu!
    software_llm = LLMChain(
        llm=FakeListLLM(
            responses=[
                "import os; print(os.getcwd())",
                "import os; os.listdir('.')",
                "print('witaj świecie!')"
            ]
        ),
        prompt=software_prompt
    )
    env = PythonDeveloper(software_llm)
