"""Document indexing.

Building a vector store fast.

Adapted from open_source_LLM_search_engine:
https://github.com/ray-project/langchain-ray/

Please note that there are FAISS versions for processing on
either CPU or GPU, which can be installed like this:
>> pip install faiss-gpu # For CUDA 7.5+ Supported GPU's.
# OR
>> pip install faiss-cpu # For CPU Installation

"""
import time

import numpy as np
import ray
from bs4 import BeautifulSoup as Soup
from langchain.vectorstores import FAISS
from langchain.document_loaders import RecursiveUrlLoader
from langchain.embeddings import OpenAIEmbeddings
from langchain.schema import Document
from langchain.text_splitter import RecursiveCharacterTextSplitter

from config import set_environment
from search_engine.utils import INDEX_PATH, get_embeddings

# set keys:
set_environment()


def chunk_docs(url: str) -> list[Document]:
    """Crawl a website and chunk the text in it.

    Wrapping the texts into list[Document] in
    order to keep the metadata.
    """
    text_splitter = RecursiveCharacterTextSplitter()

    # Wczytanie dokumentów.
    loader = RecursiveUrlLoader(
        url=url,
        max_depth=2,
        extractor=lambda x: Soup(x, "html.parser").text
    )
    docs = loader.load()

    # Podział dokumentów na zdanie za pomocą obiektu rozdzielającego z LangChain.
    return text_splitter.create_documents(
        [doc.page_content for doc in docs],
        metadatas=[doc.metadata for doc in docs]
    )


def create_db(chunks: list[Document]) -> FAISS:
    """To jest łatwy sposób."""
    return FAISS.from_documents(
        chunks,
        OpenAIEmbeddings()
        # get_embeddings()
    )


@ray.remote
def process_shard(chunks: list[Document]):
    """Przetwarzanie zadania..

    W części dekoratora możesz określić liczbę procesorów i procesorów graficznych, które chcesz wykorzystać.
    """
    return FAISS.from_documents(
        documents=chunks,
        embedding=get_embeddings()
    )


def create_db_parallel(chunks: list[Document]):
    """Create a FAISS db with parallelism."""
    # Podział fragmentów na mniejsze części:
    shards = np.array_split(chunks, 8)

    # Uruchomienie Ray.
    ray.init()

    # Równoległe przetwarzanie części:
    futures = [process_shard.remote(shard) for shard in shards]
    results = ray.get(futures)

    # Scalanie indeksów części.
    db = results[0]
    for result in results[1:]:
        db.merge_from(result)

    # Zatrzymanie Ray:
    ray.shutdown()
    return db


if __name__ == "__main__":
    print("Rozpoczęcie procesu indeksowania.")
    st = time.time()
    chunks = chunk_docs(url="https://docs.ray.io/en/latest/")
    if len(chunks) == 0:
        raise ValueError("Nie utworzono żadnych fragmentów!")
    db = create_db(chunks)  # create_db_parallel(chunks)
    db.save_local(INDEX_PATH)
    et = time.time() - st
    print(f"Ukończono w {et} sekund.")
    """
    Rozpoczęcie procesu indeksowania.
    Zakończono w 39.936267137527466 sekund.
    """
