// Nazwa pliku: ExtremeC_examples_chapter17_6.c
// Opis: Prezentacja wyścigu danych między dwoma procesami
//              podczas używania pamięci współdzielonej do
//              przeprowadzania operacji na liczniku współdzielonym.

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/wait.h>

#define SH_SIZE 4

// Współdzielony deskryptor pliku używany w celu odwołania się
// do obiektu pamięci współdzielonej.
int shared_fd = -1;

// Wskaźnik prowadzący do licznika współdzielonego.
int32_t* counter = NULL;

void init_shared_resource() {
  // Otworzenie obiektu pamięci współdzielonej.
  shared_fd = shm_open("/shm0", O_CREAT | O_RDWR, 0600);
  if (shared_fd < 0) {
    fprintf(stderr, "BŁĄD! Nie udało się utworzyć obiektu pamięci współdzielonej: %s\n",
        strerror(errno));
    exit(1);
  }
  fprintf(stdout, "Obiekt pamięci współdzielonej został utworzony z deskryptorem pliku o wartości: %d\n",
          shared_fd);
}

void shutdown_shared_resource() {
  if (shm_unlink("/shm0") < 0) {
    fprintf(stderr, "BŁĄD! Zwolnienie pamięci współdzielonej zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    exit(1);
  }
}

void inc_counter() {
  usleep(1);
  int32_t temp = *counter;
  usleep(1);
  temp++;
  usleep(1);
  *counter = temp;
  usleep(1);
}

int main(int argc, char** argv) {

  // Proces nadrzędny musi zainicjalizować zasób współdzielony.
  init_shared_resource();

  // Alokacja i zmniejszenie obszaru pamięci współdzielonej.
  if (ftruncate(shared_fd, SH_SIZE * sizeof(char)) < 0) {
    fprintf(stderr, "BŁĄD! Próba zmniejszenia obszaru pamięci zakończyła się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  fprintf(stdout, "Obszar pamięci został zmniejszony.\n");

  // Mapowanie pamięci współdzielonej i inicjalizacja wskaźnika.
  void* map = mmap(0, SH_SIZE, PROT_WRITE,
          MAP_SHARED, shared_fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "BŁĄD! Mapowanie zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  counter = (int32_t*)map;
  *counter = 0;

  // Utworzenie nowego procesu.
  pid_t pid = fork();
  if (pid) { // Proces nadrzędny.
    // Inkrementacja licznika.
    inc_counter();
    fprintf(stdout, "Dla procesu nadrzędnego licznik ma wartość %d.\n",
        *counter);

    // Oczekiwanie na zakończenie działania procesu potomnego.
    int status = -1;
    wait(&status);
    fprintf(stdout, "Proces potomny zakończył działanie i zwrócił wartość %d.\n",
        status);
  } else { // Proces potomny.
    // Inkrementacja licznika.
    inc_counter();
    fprintf(stdout, "Dla procesu potomnego licznik ma wartość %d.\n",
        *counter);
  }

  // Oba procesy powinny zakończyć mapowanie regionu pamięci współdzielonej
  // i zamknąć swoje deskryptory plików.
  if (munmap(counter, SH_SIZE) < 0) {
    fprintf(stderr, "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  if (close(shared_fd) < 0) {
    fprintf(stderr, "BŁĄD! Zamknięcie obiektu pamięci współdzielonej z deskryptorem pliku o wartości: %s\n",
        strerror(errno));
    return 1;
  }

  // Tylko proces nadrzędny musi zamknąć zasób współdzielony.
  if (pid) {
    shutdown_shared_resource();
  }

  return 0;
}
