// Nazwa pliku: stack.go
// Opis: Opakowanie Go dla utworzonej w C biblioteki stosu.

package main

/*
#cgo CFLAGS: -I..
#cgo LDFLAGS: -L.. -lcstack
#include <cstack.h>
*/
import "C"
import (
  "fmt"
)

type Stack struct {
  handler *C.cstack_t
}

func NewStack() *Stack {
  s := new(Stack)
  s.handler = C.cstack_new()
  C.cstack_ctor(s.handler, 100)
  return s
}

func (s *Stack) Destroy() {
  C.cstack_dtor(s.handler, C.deleter_t(C.free_value))
  C.cstack_delete(s.handler)
}

func (s *Stack) Size() int {
  return int(C.cstack_size(s.handler))
}

func (s *Stack) Push(item string) bool {
  value := C.make_value(C.CString(item), C.ulong(len(item) + 1))
  pushed := C.cstack_push(s.handler, value)
  return pushed == 1
}

func (s *Stack) Pop() (bool, string) {
  value := C.make_value(nil, 0)
  popped := C.cstack_pop(s.handler, &value)
  str := C.GoString(value.data)
  defer C.free_value(&value)
  return popped == 1, str
}

func (s *Stack) Clear() {
  C.cstack_clear(s.handler, C.deleter_t(C.free_value))
}

func main() {
  var stack = NewStack()
  stack.Push("Witaj,")
  stack.Push("świecie")
  stack.Push("!")
  fmt.Println("Wielkość stosu:", stack.Size())
  for stack.Size() > 0 {
    _, str := stack.Pop()
    fmt.Println("Usunięto >", str)
  }
  fmt.Println("Wielkość stosu po usunięciu obiektu:", stack.Size())
  stack.Push("Żegnaj,")
  stack.Push("świecie")
  fmt.Println("Wielkość stosu przed czyszczeniem:", stack.Size())
  stack.Clear()
  fmt.Println("Wielkość stosu po czyszczeniu:", stack.Size())
  stack.Destroy()
}
