// Nazwa pliku: ExtremeC_examples_chapter17_7.c
// Opis: Prezentacja wyścigu danych między dwoma procesami
//              podczas używania systemu plików do
//              przechowywania stanu współdzielonego.

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/wait.h>

#define SH_SIZE 4

// Współdzielony deskryptor pliku używany w celu odwołania się do pliku współdzielonego.
int shared_fd = -1;

// Wskaźnik prowadzący do licznika współdzielonego.
int32_t* counter = NULL;

void init_shared_resource() {
  // Otworzenie pliku.
  shared_fd = open("data.bin", O_CREAT | O_RDWR, 0600);
  if (shared_fd < 0) {
    fprintf(stderr, "BŁĄD! Nie udało się utworzyć pliku: %s\n",
        strerror(errno));
    exit(1);
  }
  fprintf(stdout, "Plik został utworzony i otworzony z deskryptorem pliku o wartości: %d\n", 
          shared_fd);
}

void shutdown_shared_resource() {
  if (remove("data.bin") < 0) {
    fprintf(stderr, "BŁĄD! Nie udało się usunąć pliku: %s\n",
        strerror(errno));
    exit(1);
  }
}

void inc_counter() {
  usleep(1);
  int32_t temp = *counter;
  usleep(1);
  temp++;
  usleep(1);
  *counter = temp;
  usleep(1);
}

int main(int argc, char** argv) {

  // Proces nadrzędny musi zainicjalizować zasób współdzielony.
  init_shared_resource();

  // Alokacja i zmniejszenie obszaru pamięci współdzielonej.
  if (ftruncate(shared_fd, SH_SIZE * sizeof(char)) < 0) {
    fprintf(stderr, "BŁĄD! Próba zmniejszenia obszaru pamięci zakończyła się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  fprintf(stdout, "Obszar pamięci został zmniejszony.\n");

  void* map = mmap(0, SH_SIZE, PROT_WRITE, MAP_SHARED,
          shared_fd, 0);
  if (map == MAP_FAILED) {
    fprintf(stderr, "BŁĄD! Mapowanie zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  counter = (int32_t*)map;
  *counter = 0;

  // Utworzenie nowego procesu.
  pid_t pid = fork();
  if (pid) { // Proces nadrzędny.
    // Inkrementacja licznika.
    inc_counter();
    fprintf(stdout, "Dla procesu nadrzędnego licznik ma wartość %d.\n",
        *counter);

    // Oczekiwanie na zakończenie działania procesu potomnego.
    int status = -1;
    wait(&status);
    fprintf(stdout, "Proces potomny zakończył działanie i zwrócił wartość %d.\n",
        status);
  } else { // Proces potomny.
    // Inkrementacja licznika.
    inc_counter();
    fprintf(stdout, "Dla procesu potomnego licznik ma wartość %d.\n",
        *counter);
  }

  // Oba procesy powinny zakończyć mapowanie regionu pamięci współdzielonej
  // i zamknąć swoje deskryptory plików.
  if (munmap(counter, SH_SIZE) < 0) {
    fprintf(stderr, "BŁĄD! Usunięcie mapowania zakończyło się niepowodzeniem: %s\n",
            strerror(errno));
    return 1;
  }
  if (close(shared_fd) < 0) {
    fprintf(stderr, "BŁĄD: Zamknięcie pliku zakończyło się niepowodzeniem: %s\n",
        strerror(errno));
    return 1;
  }

  // Tylko proces nadrzędny musi zamknąć zasób współdzielony.
  if (pid) {
    shutdown_shared_resource();
  }

  return 0;
}
