#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
class Book:
    def __init__(self, title, due_date):
        self.title = title
        self.due_date = due_date


print("Przykład 2")
def add_book(queue, book):
    queue.append(book)
    queue.sort(key=lambda x: x.due_date, reverse=True)

queue = []
add_book(queue, Book("Don Kichot", "2019-06-07"))
add_book(queue, Book("Frankenstein", "2019-06-05"))
add_book(queue, Book("Nędznicy", "2019-06-08"))
add_book(queue, Book("Wojna i pokój", "2019-06-03"))


print("Przykład 3")
class NoOverdueBooks(Exception):
    pass

def next_overdue_book(queue, now):
    if queue:
        book = queue[-1]
        if book.due_date < now:
            queue.pop()
            return book

    raise NoOverdueBooks


print("Przykład 4")
now = "2019-06-10"

found = next_overdue_book(queue, now)
print(found.due_date, found.title)

found = next_overdue_book(queue, now)
print(found.due_date, found.title)


print("Przykład 5")
def return_book(queue, book):
    queue.remove(book)

queue = []
book = Book("Wyspa skarbów", "2019-06-04")

add_book(queue, book)
print("Przed terminem:", [x.title for x in queue])

return_book(queue, book)
print("Po terminie: ", [x.title for x in queue])


print("Przykład 6")
try:
    next_overdue_book(queue, now)
except NoOverdueBooks:
    pass          # Oczekiwane
else:
    assert False  # Nie zdarzyło się


print("Przykład 7")
import random
import timeit

def list_overdue_benchmark(count):
    def prepare():
        to_add = list(range(count))
        random.shuffle(to_add)
        return [], to_add

    def run(queue, to_add):
        for i in to_add:
            queue.append(i)
            queue.sort(reverse=True)

        while queue:
            queue.pop()

    return timeit.timeit(
        setup="queue, to_add = prepare()",
        stmt=f"run(queue, to_add)",
        globals=locals(),
        number=1,
    )


print("Przykład 8")
for i in range(1, 6):
    count = i * 1_000
    delay = list_overdue_benchmark(count)
    print(f"Zliczenie {count:>5,} elementów zajęło {delay*1e3:>6.2f}ms")


print("Przykład 9")
def list_return_benchmark(count):
    def prepare():
        queue = list(range(count))
        random.shuffle(queue)

        to_return = list(range(count))
        random.shuffle(to_return)

        return queue, to_return

    def run(queue, to_return):
        for i in to_return:
            queue.remove(i)

    return timeit.timeit(
        setup="queue, to_return = prepare()",
        stmt=f"run(queue, to_return)",
        globals=locals(),
        number=1,
    )


print("Przykład 10")
for i in range(1, 6):
    count = i * 1_000
    delay = list_return_benchmark(count)
    print(f"Zliczenie {count:>5,} elementów zajęło {delay*1e3:>6.2f}ms")


print("Przykład 11")
from heapq import heappush

def add_book(queue, book):
    heappush(queue, book)


print("Przykład 12")
try:
    queue = []
    add_book(queue, Book("Małe kobietki", "2019-06-05"))
    add_book(queue, Book("Wehikuł czasu", "2019-05-30"))
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 13")
import functools

@functools.total_ordering
class Book:
    def __init__(self, title, due_date):
        self.title = title
        self.due_date = due_date

    def __lt__(self, other):
        return self.due_date < other.due_date


print("Przykład 14")
queue = []
add_book(queue, Book("Duma i uprzedzenie", "2019-06-01"))
add_book(queue, Book("Wehikuł czasu", "2019-05-30"))
add_book(queue, Book("Zbrodnia i kara", "2019-06-06"))
add_book(queue, Book("Wichrowe wzgórza", "2019-06-12"))
print([b.title for b in queue])


print("Przykład 15")
queue = [
    Book("Duma i uprzedzenie", "2019-06-01"),
    Book("Wehikuł czasu", "2019-05-30"),
    Book("Zbrodnia i kara", "2019-06-06"),
    Book("Wichrowe wzgórza", "2019-06-12"),
]
queue.sort()
print([b.title for b in queue])


print("Przykład 16")
from heapq import heapify

queue = [
    Book("Duma i uprzedzenie", "2019-06-01"),
    Book("Wehikuł czasu", "2019-05-30"),
    Book("Zbrodnia i kara", "2019-06-06"),
    Book("Wichrowe wzgórza", "2019-06-12"),
]
heapify(queue)
print([b.title for b in queue])


print("Przykład 17")
from heapq import heappop

def next_overdue_book(queue, now):
    if queue:
        book = queue[0]     # Pierwsza jest najdłużej przetrzymana książka
        if book.due_date < now:
            heappop(queue)  # Usunięcie przetrzymanej książki
            return book

    raise NoOverdueBooks


print("Przykład 18")
now = "2019-06-02"

book = next_overdue_book(queue, now)
print(book.due_date, book.title)

book = next_overdue_book(queue, now)
print(book.due_date, book.title)

try:
    next_overdue_book(queue, now)
except NoOverdueBooks:
    pass  # Oczekiwane
else:
    assert False  # Nie zdarzyło się


print("Przykład 19")
def heap_overdue_benchmark(count):
    def prepare():
        to_add = list(range(count))
        random.shuffle(to_add)
        return [], to_add

    def run(queue, to_add):
        for i in to_add:
            heappush(queue, i)
        while queue:
            heappop(queue)

    return timeit.timeit(
        setup="queue, to_add = prepare()",
        stmt=f"run(queue, to_add)",
        globals=locals(),
        number=1,
    )


print("Przykład 20")
for i in range(1, 6):
    count = i * 10_000
    delay = heap_overdue_benchmark(count)
    print(f"Zliczenie {count:>5,} elementów zajęło {delay*1e3:>6.2f}ms")


print("Przykład 21")
@functools.total_ordering
class Book:
    def __init__(self, title, due_date):
        self.title = title
        self.due_date = due_date
        self.returned = False  # Nowa właściwość

    def __lt__(self, other):
        return self.due_date < other.due_date


print("Przykład 22")
def next_overdue_book(queue, now):
    while queue:
        book = queue[0]
        if book.returned:
            heappop(queue)
            continue

        if book.due_date < now:
            heappop(queue)
            return book

        break

    raise NoOverdueBooks


queue = []

book = Book("Duma i uprzedzenie", "2019-06-01")
add_book(queue, book)

book = Book("Wehikuł czasu", "2019-05-30")
add_book(queue, book)
book.returned = True

book = Book("Zbrodnia i kara", "2019-06-06")
add_book(queue, book)
book.returned = True

book = Book("Wichrowe wzgórza", "2019-06-12")
add_book(queue, book)

now = "2019-06-11"

book = next_overdue_book(queue, now)
assert book.title == "Duma i uprzedzenie"

try:
    next_overdue_book(queue, now)
except NoOverdueBooks:
    pass  # Oczekiwane
else:
    assert False  # Nie zdarzyło się


print("Przykład 23")
def return_book(queue, book):
    book.returned = True


assert not book.returned
return_book(queue, book)
assert book.returned
