#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
def my_func():
    try:
        return 123
    finally:
        print("Blok finally funkcji my_func()")

print("Przed wywołaniem funkcji")
print(my_func())
print("Po wywołaniu funkcji")


print("Przykład 2")
def my_generator():
    try:
        yield 10
        yield 20
        yield 30
    finally:
        print("Blok finally funkcji my_generator()")

print("Przed wywołaniem funkcji")

for i in my_generator():
    print(i)

print("Po wywołaniu funkcji")


print("Przykład 3")
it = my_generator()
print("Przed wywołaniem funkcji")
print(next(it))
print(next(it))
print("Po wywołaniu funkcji")


print("Przykład 4")
import gc

del it
gc.collect()


print("Przykład 5")
def catching_generator():
    try:
        yield 40
        yield 50
        yield 60
    except BaseException as e:  # Przechwycenie wyjątku GeneratorExit
        print("Przechwycenie procedury obsługi", type(e), e)
        raise


print("Przykład 6")
it = catching_generator()
print("Przed wywołaniem funkcji")
print(next(it))
print(next(it))
print("Po wywołaniu funkcji")
del it
gc.collect()


print("Przykład 8")
with open("my_file.txt", "w") as f:
    for _ in range(20):
        f.write("a" * random.randint(0, 100))
        f.write("\n")

def lengths_path(path):
    try:
        with open(path) as handle:
            for i, line in enumerate(handle):
                print(f"Wiersz {i}")
                yield len(line.strip())
    finally:
        print("Blok finally funkcji lengths_path()")


print("Przykład 9")
max_head = 0
it = lengths_path("my_file.txt")

for i, length in enumerate(it):
    if i == 5:
        break
    else:
        max_head = max(max_head, length)

print(max_head)


print("Przykład 10")
del it
gc.collect()


print("Przykład 11")
def lengths_handle(handle):
    try:
        for i, line in enumerate(handle):
            print(f"Wiersz {i}")
            yield len(line.strip())
    finally:
        print("Blok finally funkcji lengths_handle()")


print("Przykład 12")
max_head = 0

with open("my_file.txt") as handle:
    it = lengths_handle(handle)
    for i, length in enumerate(it):
        if i == 5:
            break
        else:
            max_head = max(max_head, length)

print(max_head)
print("Uchwyt pliku został zamknięty:", handle.closed)
