#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
class Field:
    def __init__(self, column_name):
        self.column_name = column_name
        self.internal_name = "_" + self.column_name


    print("Przykład 2")
    def __get__(self, instance, instance_type):
        if instance is None:
            return self
        return getattr(instance, self.internal_name, "")

    def __set__(self, instance, value):
        setattr(instance, self.internal_name, value)


print("Przykład 3")
class Customer:
    # Atrybuty klasy
    first_name = Field("first_name")
    last_name = Field("last_name")
    prefix = Field("prefix")
    suffix = Field("suffix")


print("Przykład 4")
cust = Customer()
print(f"Przed: {cust.first_name!r} {cust.__dict__}")
cust.first_name = "Euclid"
print(f"Po:  {cust.first_name!r} {cust.__dict__}")


print("Przykład 5")
class Customer:
    # Lewa strona jest zbędna dla użytej tutaj prawej strony
    first_name = Field("first_name")
    last_name = Field("last_name")
    prefix = Field("prefix")
    suffix = Field("suffix")


print("Przykład 6")
class Meta(type):
    def __new__(meta, name, bases, class_dict):
        for key, value in class_dict.items():
            if isinstance(value, Field):
                value.column_name = key
                value.internal_name = "_" + key
        cls = type.__new__(meta, name, bases, class_dict)
        return cls


print("Przykład 7")
class DatabaseRow(metaclass=Meta):
    pass


print("Przykład 8")
class Field:
    def __init__(self):
        # Te przypisania zostaną wykonane przez metaklasę
        self.column_name = None
        self.internal_name = None

    def __get__(self, instance, instance_type):
        if instance is None:
            return self
        return getattr(instance, self.internal_name, "")

    def __set__(self, instance, value):
        setattr(instance, self.internal_name, value)


print("Przykład 9")
class BetterCustomer(DatabaseRow):
    first_name = Field()
    last_name = Field()
    prefix = Field()
    suffix = Field()


print("Przykład 10")
cust = BetterCustomer()
print(f"Przed: {cust.first_name!r} {cust.__dict__}")
cust.first_name = "Euler"
print(f"Po:  {cust.first_name!r} {cust.__dict__}")


print("Przykład 11")
try:
    class BrokenCustomer:  # Brak informacji o dziedziczeniu
        first_name = Field()
        last_name = Field()
        prefix = Field()
        suffix = Field()
    
    cust = BrokenCustomer()
    cust.first_name = "Mersenne"
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 12")
class Field:
    def __init__(self):
        self.column_name = None
        self.internal_name = None

    def __set_name__(self, owner, column_name):
        # Podczas tworzenia klasy ta metoda wywoływana dla każdego deskryptora
        self.column_name = column_name
        self.internal_name = "_" + column_name

    def __get__(self, instance, instance_type):
        if instance is None:
            return self
        return getattr(instance, self.internal_name, "")

    def __set__(self, instance, value):
        setattr(instance, self.internal_name, value)


print("Przykład 13")
class FixedCustomer:  # Brak klasy nadrzędnej
    first_name = Field()
    last_name = Field()
    prefix = Field()
    suffix = Field()

cust = FixedCustomer()
print(f"Przed: {cust.first_name!r} {cust.__dict__}")
cust.first_name = "Mersenne"
print(f"Po:  {cust.first_name!r} {cust.__dict__}")
