#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
names = ["Sokrates", "Archimedes", "Platon", "Arystoteles"]
names.sort(key=len)
print(names)


print("Przykład 2")
def log_missing():
    print("Klucz został dodany")
    return 0


print("Przykład 3")
from collections import defaultdict

current = {"zielony": 12, "niebieski": 3}
increments = [
    ("czerwony", 5),
    ("niebieski", 17),
    ("pomarańczowy", 9),
]
result = defaultdict(log_missing, current)
print("Przed:", dict(result))
for key, amount in increments:
    result[key] += amount
print("Po: ", dict(result))


print("Przykład 4")
def increment_with_report(current, increments):
    added_count = 0

    def missing():
        nonlocal added_count  # Domknięcie wraz z informacjami o stanie
        added_count += 1
        return 0

    result = defaultdict(missing, current)
    for key, amount in increments:
        result[key] += amount

    return result, added_count


print("Przykład 5")
result, count = increment_with_report(current, increments)
assert count == 2
print(result)


print("Przykład 6")
class CountMissing:
    def __init__(self):
        self.added = 0

    def missing(self):
        self.added += 1
        return 0


print("Przykład 7")
counter = CountMissing()
result = defaultdict(counter.missing, current)  # Odwołanie do metody
for key, amount in increments:
    result[key] += amount
assert counter.added == 2
print(result)


print("Przykład 8")
class BetterCountMissing:
    def __init__(self):
        self.added = 0

    def __call__(self):
        self.added += 1
        return 0

counter = BetterCountMissing()
assert counter() == 0
assert callable(counter)


print("Przykład 9")
counter = BetterCountMissing()
result = defaultdict(counter, current)  # Oparcie na metodzie __call__()
for key, amount in increments:
    result[key] += amount
assert counter.added == 2
print(result)
