#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
class GameState:
    def __init__(self):
        self.level = 0
        self.lives = 4


print("Przykład 2")
state = GameState()
state.level += 1  # Gracz zakończył poziom
state.lives -= 1  # Gracz musi spróbować ponownie przejść dany poziom

print(state.__dict__)


print("Przykład 3")
import pickle

state_path = "game_state.bin"
with open(state_path, "wb") as f:
    pickle.dump(state, f)


print("Przykład 4")
with open(state_path, "rb") as f:
    state_after = pickle.load(f)

print(state_after.__dict__)


print("Przykład 5")
class GameState:
    def __init__(self):
        self.level = 0
        self.lives = 4
        self.points = 0  # Nowa właściwość


print("Przykład 6")
state = GameState()
serialized = pickle.dumps(state)
state_after = pickle.loads(serialized)

print(state_after.__dict__)


print("Przykład 7")
with open(state_path, "rb") as f:
    state_after = pickle.load(f)

print(state_after.__dict__)


print("Przykład 8")
assert isinstance(state_after, GameState)


print("Przykład 9")
class GameState:
    def __init__(self, level=0, lives=4, points=0):
        self.level = level
        self.lives = lives
        self.points = points


print("Przykład 10")
def pickle_game_state(game_state):
    kwargs = game_state.__dict__
    return unpickle_game_state, (kwargs,)


print("Przykład 11")
def unpickle_game_state(kwargs):
    return GameState(**kwargs)


print("Przykład 12")
import copyreg

copyreg.pickle(GameState, pickle_game_state)


print("Przykład 13")
state = GameState()
state.points += 1000
serialized = pickle.dumps(state)
state_after = pickle.loads(serialized)
print(state_after.__dict__)


print("Przykład 14")
class GameState:
    def __init__(self, level=0, lives=4, points=0, magic=5):
        self.level = level
        self.lives = lives
        self.points = points
        self.magic = magic  # Nowa właściwość


print("Przykład 15")
print("Przed:", state.__dict__)
state_after = pickle.loads(serialized)
print("Po: ", state_after.__dict__)


print("Przykład 16")
class GameState:
    def __init__(self, level=0, points=0, magic=5):
        self.level = level
        self.points = points
        self.magic = magic


print("Przykład 17")
try:
    pickle.loads(serialized)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 18")
def pickle_game_state(game_state):
    kwargs = game_state.__dict__
    kwargs["version"] = 2
    return unpickle_game_state, (kwargs,)


print("Przykład 19")
def unpickle_game_state(kwargs):
    version = kwargs.pop("version", 1)
    if version == 1:
        del kwargs["lives"]
    return GameState(**kwargs)


print("Przykład 20")
copyreg.pickle(GameState, pickle_game_state)
print("Przed:", state.__dict__)
state_after = pickle.loads(serialized)
print("Po: ", state_after.__dict__)


print("Przykład 21")
copyreg.dispatch_table.clear()
state = GameState()
serialized = pickle.dumps(state)
del GameState

class BetterGameState:
    def __init__(self, level=0, points=0, magic=5):
        self.level = level
        self.points = points
        self.magic = magic


print("Przykład 22")
try:
    pickle.loads(serialized)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 23")
print(serialized)


print("Przykład 24")
copyreg.pickle(BetterGameState, pickle_game_state)


print("Przykład 25")
state = BetterGameState()
serialized = pickle.dumps(state)
print(serialized)
