#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
class LazyRecord:
    def __init__(self):
        self.exists = 5

    def __getattr__(self, name):
        value = f"Wartość dla {name}"
        setattr(self, name, value)
        return value


print("Przykład 2")
data = LazyRecord()
print("Przed:", data.__dict__)
print("foo:   ", data.foo)
print("Po: ", data.__dict__)


print("Przykład 3")
class LoggingLazyRecord(LazyRecord):
    def __getattr__(self, name):
        print(
            f"* Wywołano __getattr__({name!r}), "
            f"wypełniono słownik egzemplarza"
        )
        result = super().__getattr__(name)
        print(f"* Zwrócono {result!r}")
        return result

data = LoggingLazyRecord()
print("exists:     ", data.exists)
print("Pierwsze foo:  ", data.foo)
print("Drugie foo: ", data.foo)


print("Przykład 4")
class ValidatingRecord:
    def __init__(self):
        self.exists = 5

    def __getattribute__(self, name):
        print(f"* Wywołano __getattribute__({name!r})")
        try:
            value = super().__getattribute__(name)
            print(f"* Znaleziono {name!r}, returning {value!r}")
            return value
        except AttributeError:
            value = f"Wartość dla {name}"
            print(f"* {name!r} przypisano wartość {value!r}")
            setattr(self, name, value)
            return value

data = ValidatingRecord()
print("exists:     ", data.exists)
print("Pierwsze foo:  ", data.foo)
print("Drugie foo: ", data.foo)


print("Przykład 5")
try:
    class MissingPropertyRecord:
        def __getattr__(self, name):
            if name == "bad_name":
                raise AttributeError(f"{name} nie istnieje")
            value = f"Wartość dla {name}"
            setattr(self, name, value)
            return value
    
    data = MissingPropertyRecord()
    assert data.foo == "Wartość dla foo"  # Sprawdzenie działania
    data.bad_name
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 6")
data = LoggingLazyRecord()  # Implementuje __getattr__
print("Przed:         ", data.__dict__)
print("Pierwsze foo istnieje:  ", hasattr(data, "foo"))
print("Po:          ", data.__dict__)
print("Drugie foo istnieje: ", hasattr(data, "foo"))


print("Przykład 7")
data = ValidatingRecord()  # Implementuje __getattribute__
print("Pierwsze foo istnieje:  ", hasattr(data, "foo"))
print("Drugie foo istnieje: ", hasattr(data, "foo"))


print("Przykład 8")
class SavingRecord:
    def __setattr__(self, name, value):
        # Zapis pewnych danych w dzienniku zdarzeń bazy danych
        pass
        super().__setattr__(name, value)


print("Przykład 9")
class LoggingSavingRecord(SavingRecord):
    def __setattr__(self, name, value):
        print(f"* Wywołano __setattr__({name!r}, {value!r})")
        super().__setattr__(name, value)

data = LoggingSavingRecord()
print("Przed: ", data.__dict__)
data.foo = 5
print("Po:  ", data.__dict__)
data.foo = 7
print("Ostatecznie:", data.__dict__)


print("Przykład 10")
class BrokenDictionaryRecord:
    def __init__(self, data):
        self._data = data

    def __getattribute__(self, name):
        print(f"* Wywołano __getattribute__({name!r})")
        return self._data[name]


print("Przykład 11")
try:
    import sys
    
    sys.setrecursionlimit(50)
    data = BrokenDictionaryRecord({"foo": 3})
    data.foo
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 12")
class DictionaryRecord:
    def __init__(self, data):
        self._data = data

    def __getattribute__(self, name):
        # Uniemożliwienie dziwnych operacji z isinstance()
        # używaną w przykładowym kodzie.
        if name == "__class__":
            return DictionaryRecord
        print(f"* Wywołano __getattribute__({name!r})")
        data_dict = super().__getattribute__("_data")
        return data_dict[name]

data = DictionaryRecord({"foo": 3})
print("foo: ", data.foo)
