#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
class MyObject:
    def __init__(self):
        self.public_field = 5
        self.__private_field = 10

    def get_private_field(self):
        return self.__private_field


print("Przykład 2")
foo = MyObject()
assert foo.public_field == 5


print("Przykład 3")
assert foo.get_private_field() == 10


print("Przykład 4")
try:
    foo.__private_field
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 5")
class MyOtherObject:
    def __init__(self):
        self.__private_field = 71

    @classmethod
    def get_private_field_of_instance(cls, instance):
        return instance.__private_field

bar = MyOtherObject()
assert MyOtherObject.get_private_field_of_instance(bar) == 71


print("Przykład 6")
try:
    class MyParentObject:
        def __init__(self):
            self.__private_field = 71
    
    class MyChildObject(MyParentObject):
        def get_private_field(self):
            return self.__private_field
    
    baz = MyChildObject()
    baz.get_private_field()
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 7")
assert baz._MyParentObject__private_field == 71


print("Przykład 8")
print(baz.__dict__)


print("Przykład 9")
class MyStringClass:
    def __init__(self, value):
        self.__value = value

    def get_value(self):
        return str(self.__value)

foo = MyStringClass(5)
assert foo.get_value() == "5"


print("Przykład 10")
class MyIntegerSubclass(MyStringClass):
    def get_value(self):
        return int(self._MyStringClass__value)

foo = MyIntegerSubclass("5")
assert foo.get_value() == 5


print("Przykład 11")
class MyBaseClass:
    def __init__(self, value):
        self.__value = value

    def get_value(self):
        return self.__value

class MyStringClass(MyBaseClass):
    def get_value(self):
        return str(super().get_value())         # Uaktualniona

class MyIntegerSubclass(MyStringClass):
    def get_value(self):
        return int(self._MyStringClass__value)  # Nieuaktualniona


print("Przykład 12")
try:
    foo = MyIntegerSubclass(5)
    foo.get_value()
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 13")
class MyStringClass:
    def __init__(self, value):
        # Kod przechowuje podaną przez użytkownika wartość dla obiektu
        # To powinien być ciąg tekstowy. Po przypisaniu obiektowi
        # powinien być traktowany jako niemodyfikowalny.
        self._value = value


    def get_value(self):
        return str(self._value)


class MyIntegerSubclass(MyStringClass):
    def get_value(self):
        return self._value

foo = MyIntegerSubclass(5)
assert foo.get_value() == 5


print("Przykład 14")
class ApiClass:
    def __init__(self):
        self._value = 5

    def get(self):
        return self._value

class Child(ApiClass):
    def __init__(self):
        super().__init__()
        self._value = "witaj"  # Konflikty

a = Child()
print(f"{a.get()} i {a._value} powinny być różnymi wartościami")


print("Przykład 15")
class ApiClass:
    def __init__(self):
        self.__value = 5     # Dwa znaki podkreślenia

    def get(self):
        return self.__value  # Dwa znaki podkreślenia

class Child(ApiClass):
    def __init__(self):
        super().__init__()
        self._value = "witaj"  # OK!

a = Child()
print(f"{a.get()} i {a._value} są różnymi wartościami")
