#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
try:
    class MyError(Exception):
        pass
    
    def my_generator():
        yield 1
        yield 2
        yield 3
    
    it = my_generator()
    print(next(it))                         # Polecenie yield 1
    print(next(it))                         # Polecenie yield 2
    print(it.throw(MyError("przykładowy błąd")))  # Zgłoszenie błędu
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 2")
def my_generator():
    yield 1

    try:
        yield 2
    except MyError:
        print("Otrzymano egzemplarz MyError!")
    else:
        yield 3

    yield 4

it = my_generator()
print(next(it))                         # Polecenie yield 1
print(next(it))                         # Polecenie yield 2
print(it.throw(MyError("przykładowy błąd")))  # Polecenie yield 4


print("Przykład 3")
class Reset(Exception):
    pass

def timer(period):
    current = period
    while current:
        try:
            yield current
        except Reset:
            print("Zerowanie licznika")
            current = period
        else:
            current -= 1


print("Przykład 4")
ORIGINAL_RESETS = [
    False,
    False,
    False,
    True,
    False,
    True,
    False,
    False,
    False,
    False,
    False,
    False,
    False,
    False,
]
RESETS = ORIGINAL_RESETS[:]

def check_for_reset():
    # Pobranie zewnętrznego zdarzenia
    return RESETS.pop(0)

def announce(remaining):
    print(f"Liczba pozostałych tyknięć: {remaining}")

def run():
    it = timer(4)
    while True:
        try:
            if check_for_reset():
                current = it.throw(Reset())
            else:
                current = next(it)
        except StopIteration:
            break
        else:
            announce(current)

run()


print("Przykład 5")
class Timer:
    def __init__(self, period):
        self.current = period
        self.period = period

    def reset(self):
        print("Zerowanie licznika")
        self.current = self.period

    def tick(self):
        before = self.current
        self.current -= 1
        return before

    def __bool__(self):
        return self.current > 0


print("Przykład 6")
RESETS = ORIGINAL_RESETS[:]

def run():
    timer = Timer(4)
    while timer:
        if check_for_reset():
            timer.reset()

        announce(timer.tick())

run()
