#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
def safe_division(
    number,
    divisor,
    ignore_overflow,
    ignore_zero_division,
):
    try:
        return number / divisor
    except OverflowError:
        if ignore_overflow:
            return 0
        else:
            raise
    except ZeroDivisionError:
        if ignore_zero_division:
            return float("inf")
        else:
            raise


print("Przykład 2")
result = safe_division(1.0, 10**500, True, False)
print(result)


print("Przykład 3")
result = safe_division(1.0, 0, False, True)
print(result)


print("Przykład 4")
def safe_division_b(
    number,
    divisor,
    ignore_overflow=False,       # Zmiana
    ignore_zero_division=False,  # Zmiana
):
    try:
        return number / divisor
    except OverflowError:
        if ignore_overflow:
            return 0
        else:
            raise
    except ZeroDivisionError:
        if ignore_zero_division:
            return float("inf")
        else:
            raise


print("Przykład 5")
result = safe_division_b(1.0, 10**500, ignore_overflow=True)
print(result)

result = safe_division_b(1.0, 0, ignore_zero_division=True)
print(result)


print("Przykład 6")
assert safe_division_b(1.0, 10**500, True, False) == 0


print("Przykład 7")
def safe_division_c(
    number,
    divisor,
    *,  # Nowe polecenie
    ignore_overflow=False,
    ignore_zero_division=False,
):
    try:
        return number / divisor
    except OverflowError:
        if ignore_overflow:
            return 0
        else:
            raise
    except ZeroDivisionError:
        if ignore_zero_division:
            return float("inf")
        else:
            raise


print("Przykład 8")
try:
    safe_division_c(1.0, 10**500, True, False)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 9")
result = safe_division_c(1.0, 0, ignore_zero_division=True)
assert result == float("inf")

try:
    result = safe_division_c(1.0, 0)
except ZeroDivisionError:
    pass  # Oczekiwane
else:
    assert False


print("Przykład 10")
assert safe_division_c(number=2, divisor=5) == 0.4
assert safe_division_c(divisor=5, number=2) == 0.4
assert safe_division_c(2, divisor=5) == 0.4


print("Przykład 11")
def safe_division_d(
    numerator,    # Zmiana
    denominator,  # Zmiana
    *,
    ignore_overflow=False,
    ignore_zero_division=False
):
    try:
        return numerator / denominator
    except OverflowError:
        if ignore_overflow:
            return 0
        else:
            raise
    except ZeroDivisionError:
        if ignore_zero_division:
            return float("inf")
        else:
            raise


print("Przykład 12")
try:
    safe_division_d(number=2, divisor=5)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 13")
def safe_division_e(
    numerator,
    denominator,
    /,  # Nowe polecenie
    *,
    ignore_overflow=False,
    ignore_zero_division=False,
):
    try:
        return numerator / denominator
    except OverflowError:
        if ignore_overflow:
            return 0
        else:
            raise
    except ZeroDivisionError:
        if ignore_zero_division:
            return float("inf")
        else:
            raise


print("Przykład 14")
assert safe_division_e(2, 5) == 0.4


print("Przykład 15")
try:
    safe_division_e(numerator=2, denominator=5)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 16")
def safe_division_f(
    numerator,
    denominator,
    /,
    ndigits=10,  # Zmiana
    *,
    ignore_overflow=False,
    ignore_zero_division=False,
):
    try:
        fraction = numerator / denominator  # Zmiana
        return round(fraction, ndigits)     # Zmiana
    except OverflowError:
        if ignore_overflow:
            return 0
        else:
            raise
    except ZeroDivisionError:
        if ignore_zero_division:
            return float("inf")
        else:
            raise


print("Przykład 17")
result = safe_division_f(22, 7)
print(result)

result = safe_division_f(22, 7, 5)
print(result)

result = safe_division_f(22, 7, ndigits=2)
print(result)
