#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
def careful_divide(a, b):
    try:
        return a / b
    except ZeroDivisionError:
        return None

assert careful_divide(4, 2) == 2
assert careful_divide(0, 1) == 0
assert careful_divide(3, 6) == 0.5
assert careful_divide(1, 0) == None


# Przykład 2.
x, y = 1, 0
result = careful_divide(x, y)
if result is None:
    print('Nieprawidłowe dane wejściowe.')
else:
    print('Wynik operacji to %.1f' % result)


# Przykład 3.
x, y = 0, 5
result = careful_divide(x, y)
if not result:
    print('Nieprawidłowe dane wejściowe.')  # To jest błędne podejście!
else:
    assert False


# Przykład 4.
def careful_divide(a, b):
    try:
        return True, a / b
    except ZeroDivisionError:
        return False, None

assert careful_divide(4, 2) == (True, 2)
assert careful_divide(0, 1) == (True, 0)
assert careful_divide(3, 6) == (True, 0.5)
assert careful_divide(1, 0) == (False, None)


# Przykład 5.
x, y = 5, 0
success, result = careful_divide(x, y)
if not success:
    print('Nieprawidłowe dane wejściowe.')


# Przykład 6.
x, y = 5, 0
_, result = careful_divide(x, y)
if not result:
    print('Nieprawidłowe dane wejściowe.')


# Przykład 7.
def careful_divide(a, b):
    try:
        return a / b
    except ZeroDivisionError as e:
        raise ValueError('Nieprawidłowe dane wejściowe.')


# Przykład 8.
x, y = 5, 2
try:
    result = careful_divide(x, y)
except ValueError:
    print('Nieprawidłowe dane wejściowe.')
else:
    print('Wynik operacji to %.1f' % result)


# Przykład 9.
def careful_divide(a: float, b: float) -> float:
    """Dzieli wartość a przez b.

    Zgłasza:
        ValueError: gdy nie można przeprowadzić operacji na danych wejściowych.
    """
    try:
        return a / b
    except ZeroDivisionError as e:
        raise ValueError('Nieprawidłowe dane wejściowe.')

try:
    result = careful_divide(1, 0)
    assert False
except ValueError:
    pass # Wystąpił błąd w kodzie wywołującym.

assert careful_divide(1, 5) == 0.2
