#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
from threading import Lock

lock = Lock()
with lock:
    # Operacja przeprowadzana po nałożeniu blokady.
    pass


# Przykład 2.
lock.acquire()
try:
    # Operacja przeprowadzana po nałożeniu blokady.
    pass
finally:
    lock.release()


# Przykład 3.
import logging
logging.getLogger().setLevel(logging.WARNING)

def my_function():
    logging.debug('Pewne dane debugowania')
    logging.error('Miejsce na informacje o błędzie)
    logging.debug('Jeszcze więcej danych debugowania')


# Przykład 4.
my_function()


# Przykład 5.
from contextlib import contextmanager

@contextmanager
def debug_logging(level):
    logger = logging.getLogger()
    old_level = logger.getEffectiveLevel()
    logger.setLevel(level)
    try:
        yield
    finally:
        logger.setLevel(old_level)


# Przykład 6.
with debug_logging(logging.DEBUG):
    print('* Wewnątrz:')
    my_function()

print('* Po:')
my_function()


# Przykład 7.
with open('my_output.txt', 'w') as handle:
    handle.write('To są pewne dane!')


# Przykład 8.
@contextmanager
def log_level(level, name):
    logger = logging.getLogger(name)
    old_level = logger.getEffectiveLevel()
    logger.setLevel(level)
    try:
        yield logger
    finally:
        logger.setLevel(old_level)


# Przykład 9.
with log_level(logging.DEBUG, 'my-log') as logger:
    logger.debug(f'To jest komunikat dla komponentu {logger.name}!')
    logging.debug('To nie zostanie wyświetlone')


# Przykład 10.
logger = logging.getLogger('my-log')
logger.debug('Informacje debugowania nie będą wyświetlane')
logger.error('Komunikat o błędzie zawsze będzie wyświetlony')


# Przykład 11.
with log_level(logging.DEBUG, 'other-log') as logger:
    logger.debug(f'To jest komunikat dla komponentu {logger.name}!')
    logging.debug('To nie zostanie wyświetlone')
